## Shearwater Nesting Success Analyses ##
library(ggplot2)
library(scales)
library(tidyr)
library(plyr)
library(dplyr)
library(ggpubr)
library(AICcmodavg)
library(reshape)
library(ResourceSelection)
library(lmtest)

##Load Data

wtsh.success<-read.csv("...WTSH_Raw_Data.csv", stringsAsFactors=FALSE)
wtsh.success$Dist.Path = as.numeric(wtsh.success$Dist.Path)


## Descriptive plots of nest-site Characteristics


# Total Percent Cover

figure2 = ggplot(wtsh.success, aes(x = as.factor(Site), y = Total.Cover)) + 
  geom_boxplot(fill = "white", alpha = 0.2) +
  xlab("Site") + 
  ylab("Vegetation Percent Cover") +
  scale_y_continuous(labels = percent, limits = c(0,1),expand = c(0, 0)) + 
  scale_x_discrete(name = 'Site',
                    labels = c('KBP' = 'Human-exposed', 'MCB' = 'Non-human-exposed')) +
  stat_summary(fun = mean, geom = "point", shape = 4, size = 2, color = "black", fill = "red") +
  theme(text = element_text(family = "serif"),
        axis.text.x =element_text(color = "black", size = 12),
        axis.text.y =element_text(color = "black", size = 12),
        panel.background = element_rect(fill ="white"),
        panel.grid.minor = element_blank(),
        panel.grid.major = element_blank(),
        axis.line = element_line(color = "black", size = 1),
        legend.position = "none",
        plot.title = element_text(lineheight = .8, face = "bold"))
figure2


# Nest Type

countData = wtsh.success %>% group_by(Site) %>% count(Nest.Type)
countData

figure2.5 = ggplot(countData, aes(fill = Nest.Type, y = n, x = Site)) +
  geom_bar(position="fill", stat = "identity") +
  geom_text(aes(label = n), size = 4, position = position_fill(vjust=0.5))+
  ylab("Percentage of Nests") +
  scale_x_discrete(name = 'Site',
                   labels = c('KBP' = 'Human-exposed', 'MCB' = 'Non-human-exposed')) +
  scale_y_continuous(labels = percent, expand = c(0, 0))+
  theme(text = element_text(family = "serif"),
        axis.text.x =element_text(color = "black", size = 12),
        axis.text.y =element_text(color = "black", size = 12),
        panel.background = element_rect(fill ="white"),
        panel.grid.minor = element_blank(),
        panel.grid.major = element_blank(),
        axis.line = element_line(color = "black", size = 1),
        legend.position = "bottom",
        plot.title = element_text(lineheight = .8, face = "bold")) +
  scale_fill_discrete(name = "Nest Type", labels = c("Burrow", "Indentation"))
figure2.5  

ggarrange(figure2, figure2.5, ncol = 2, nrow = 1) 


## Between site nesting success and nest-site characteristics

# Arcsine transform total percent vegetation cover and naupaka percent cover
wtsh.success$Trans.Total.Cover = asin(sqrt(wtsh.success$Total.Cover))
wtsh.success$Trans.Naupaka.Cover = asin(sqrt(wtsh.success$Naupaka.Cover))
str(wtsh.success)

# Glm models + AICc table

glm.null = glm(Success ~ 1, data = wtsh.success, family = binomial)

glm.site = glm(Success ~ Site, data = wtsh.success, family = binomial)

glm.all = glm(Success ~ Site + Trans.Total.Cover + Nest.Type, data = wtsh.success, family = binomial)

glm.wo.totcov = glm(Success ~ Site + Nest.Type, data = wtsh.success, family = binomial)

glm.wo.bur = glm(Success ~ Site + Trans.Total.Cover, data = wtsh.success, family = binomial)

glm.wo.site = glm(Success ~ Nest.Type + Trans.Total.Cover, data = wtsh.success, family = binomial)

glm.veg.wo.site = glm(Success ~ Trans.Total.Cover, data = wtsh.success, family = binomial)

glm.bur.wo.site = glm(Success ~ Nest.Type, data = wtsh.success, family = binomial)

aictab(list(glm.all, glm.null, glm.site, glm.wo.totcov, glm.wo.bur, glm.wo.site, glm.veg.wo.site, glm.bur.wo.site), 
       modnames = c("glm.all", "glm.null", "glm.site", "glm.wo.totcov", "glm.wo.bur", "glm.wo.site", "glm.veg.wo.site","glm.bur.wo.site"))
summary(glm.site)

## Hosmer-Lemeshow Goodness of Fit
hoslem.test(glm.site$y,glm.site$fitted)

##Likelihood Ratio
lrtest(glm.site)


## Nesting Success Estimates and Plotting 

# Glm w/ confidence intervals
conf.glm = glm (Success~ 0 + Site, data = wtsh.success, family = binomial)
# Estimate + Confidence interval estimation
(conf.glm1 = confint(conf.glm))
(bt.cf = exp(conf.glm1)/(1+exp(conf.glm1)))
(point.est = conf.glm$coefficients)
(bt.point.est = exp(point.est)/(1+exp(point.est)))

# Bar chart of estimated nesting success and confidence intervals

(bc.wtsh = data.frame(site = c('Human-exposed', 'Non-human-exposed'), bt.point.est, bt.cf))
colnames(bc.wtsh) = c('site', 'est', 'lower', 'upper')
rownames(bc.wtsh) = c()

figure3 = ggplot(data = bc.wtsh, aes(y = est, x = site, fill = site)) + 
  geom_bar(stat = 'identity') +
  scale_x_discrete( name = 'Site',
                    labels = c('Human-exposed' = 'Human-exposed', 'Non-human-exposed' = 'Non-human-exposed')) +
  scale_fill_manual(values = c('Human-exposed'= "grey", 'Non-human-exposed'= "grey")) +
  ylab('Probability of Nest Success') +
  geom_errorbar( aes(x = site, ymin = lower, ymax = upper), 
                 width = .2, color = "black", alpha = 0.9) + scale_y_continuous(expand = c(0, 0), limits = c(0, 1)) +
  theme(text = element_text(family = "serif"),
        legend.position = 'none', 
        axis.line = element_line(color = "black"), 
        panel.grid.major = element_blank(), 
        panel.grid.minor = element_blank(), 
        panel.background = element_blank(),
        axis.text = element_text(color = "black", size = 12, face = "plain"),
        axis.title = element_text(color = "black", size = 12, face = "plain"))
figure3
ggsave("Figure_3.jpeg", dpi = 300)


# Power Analysis
power.prop.test(n=25, p1 = 0.64, p2 = 0.83)
power.prop.test(power = 0.8, p1 = 0.64, p2=0.83)


## Human-exposed nest-site characteristic + Human disturbance models

# Subset by site
human.ex<-subset(wtsh.success, Site=="KBP")
human.ex = na.omit(human.ex)
human.ex
mean(human.ex$Human.Detections)
sd(human.ex$Human.Detections)
min(human.ex$Human.Detections)
max(human.ex$Human.Detections)

# Glm models + AICc Table

hex.glm.null = glm(Success ~ 1, data = human.ex, family = binomial)

hex.glm.all = glm(Success ~ Human.Detections + Dist.Path + Trans.Naupaka.Cover + Nest.Type, data = human.ex, family = binomial)

hex.glm.human = glm(Success ~ Human.Detections + Dist.Path, data = human.ex, family = binomial)

hex.glm.path = glm(Success ~ Dist.Path, data = human.ex, family = binomial)

hex.glm.det = glm(Success ~ Human.Detections, data = human.ex, family = binomial)

hex.glm.nau = glm(Success ~ Trans.Naupaka.Cover, data = human.ex, family = binomial)

hex.glm.det.nau = glm(Success ~ Trans.Naupaka.Cover + Human.Detections, data = human.ex, family = binomial)

hex.glm.path.nau = glm(Success ~ Trans.Naupaka.Cover + Dist.Path, data = human.ex, family = binomial)

hex.glm.nety = glm(Success ~ Nest.Type, data = human.ex, family = binomial)

hex.glm.nt.dist = glm(Success ~ Nest.Type + Dist.Path, data = human.ex, family = binomial)
  
hex.glm.nt.nau = glm(Success ~ Nest.Type + Trans.Naupaka.Cover, data = human.ex, family = binomial)  

aictab(list(hex.glm.null, hex.glm.all, hex.glm.human, hex.glm.path, 
            hex.glm.det, hex.glm.nau, hex.glm.det.nau, hex.glm.path.nau, hex.glm.nety,
            hex.glm.nt.dist, hex.glm.nt.nau),
       modnames = c("hex.glm.null", "hex.glm.all", "hex.glm.human", 
                    "hex.glm.path", "hex.glm.det", "hex.glm.nau", "hex.glm.det.nau", 
                    "hex.glm.path.nau", "hex.glm.nety", "hex.glm.nt.dist", 
                    "hex.glm.nt.nau"))