from flask import Flask, jsonify, request
from flask_cors import CORS
from utils import run_dijkstra, generate_path_image_from_db, load_model_from_db, load_nodes_from_content, find_nearest_lift
from config import SessionLocal
from models import FileStorage
import io
from datetime import datetime as dt

app = Flask(__name__)
CORS(app)

@app.route('/api/nodes', methods=['GET'])
def get_nodes():
    """API to return all nodes from all floors."""
    db = SessionLocal()

    # Query all model files to get all floors
    model_files = db.query(FileStorage).filter(FileStorage.filename.like("model-%.txt")).order_by(FileStorage.timestamp.desc()).all()
    
    all_nodes = {}

    # Load nodes for each floor from the model files
    for model_file in model_files:
        # Extract floor name from filename (e.g., "model-1.txt" -> floor_name = "1")
        floor_name = model_file.filename.split('-')[1].split('.')[0]
        nodes, _ = load_nodes_from_content(model_file.content)
        
        # Add nodes for the current floor
        all_nodes[floor_name] = nodes

    db.close()
    return jsonify(all_nodes)
    
@app.route('/api/path', methods=['POST'])
def get_path():
    """API to get the Dijkstra path between start and end nodes, with multi-floor handling."""
    data = request.get_json()
    start_node = data['start']
    end_node = data['end']
    start_floor = data['start_floor']
    end_floor = data['end_floor']

    print(f"Request received for pathfinding from '{start_node}' on floor '{start_floor}' to '{end_node}' on floor '{end_floor}'.")

    # Single-floor pathfinding
    if start_floor == end_floor:
        nodes, paths = load_model_from_db(start_floor)
        path = run_dijkstra(start_node, end_node, nodes, paths)
        if not path:
            path = run_dijkstra(end_node, start_node, nodes, paths)
            if path:
                path = path[::-1]
        if path:
            img_base64 = generate_path_image_from_db(path, nodes, start_floor)
            return jsonify({"image": img_base64, "floor": start_floor})

    # Multi-floor pathfinding with nearest lift
    nodes_start, paths_start = load_model_from_db(start_floor)
    nodes_end, paths_end = load_model_from_db(end_floor)

    # Find nearest lift on the start floor
    nearest_lift_start = find_nearest_lift(start_node, nodes_start)
    nearest_lift_end = nearest_lift_start  # Use the same lift on the end floor

    # Path from start to nearest lift on start floor
    path_to_lift = run_dijkstra(start_node, nearest_lift_start, nodes_start, paths_start)
    if not path_to_lift:
        path_to_lift = run_dijkstra(nearest_lift_start, start_node, nodes_start, paths_start)
        if path_to_lift:
            path_to_lift = path_to_lift[::-1]

    # Path from lift on end floor to end node
    path_from_lift = run_dijkstra(nearest_lift_end, end_node, nodes_end, paths_end)
    if not path_from_lift:
        path_from_lift = run_dijkstra(end_node, nearest_lift_end, nodes_end, paths_end)
        if path_from_lift:
            path_from_lift = path_from_lift[::-1]

    # Final response if paths are found
    if path_to_lift and path_from_lift:
        img_base64_start = generate_path_image_from_db(path_to_lift, nodes_start, start_floor)
        img_base64_end = generate_path_image_from_db(path_from_lift, nodes_end, end_floor)
        return jsonify({
            "start_floor": {"image": img_base64_start, "floor": start_floor},
            "end_floor": {"image": img_base64_end, "floor": end_floor}
        })

    return jsonify({"error": "Path does not exist"}), 404


@app.route('/api/update_file', methods=['POST'])
def update_file():
    """API to update or add a new version of a file for a specific floor."""
    db = SessionLocal()
    file = request.files.get('file')
    
    if not file:
        return jsonify({"error": "No file provided"}), 400
    
    filename = file.filename
    content = file.read()
    file_type = 'image' if filename.endswith('.png') else 'text'

    try:
        # Add the new file version to the database
        new_file = FileStorage(
            filename=filename,
            file_type=file_type,
            content=content,
            timestamp=dt.utcnow()  # Correct usage of datetime for timestamp
        )
        db.add(new_file)
        db.commit()
        db.close()
        
        return jsonify({"status": "File updated successfully"}), 200
    except Exception as e:
        db.rollback()
        print("Error saving file to database:", e)
        return jsonify({"error": "Failed to save file"}), 500

if __name__ == '__main__':
    app.run(debug=True)
