'''Train CIFAR10 with PyTorch.'''
import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.backends.cudnn as cudnn

import torchvision
import torchvision.transforms as transforms

import os
import argparse
from models import *
from utils import progress_bar, get_logger
import thop
from thop import profile
from thop import clever_format
import datetime
parser = argparse.ArgumentParser(description='PyTorch CIFAR10 Training')
parser.add_argument('--lr', default=0.1, type=float, help='learning rate')
parser.add_argument('--target_ratio', default=0.5, type=float, help='target prune rate')
parser.add_argument('--resume', '-r', action='store_true',
                    help='resume from checkpoint')
args = parser.parse_args()
now = datetime.datetime.now().strftime('%Y-%m-%d-%H:%M:%S')
logger = get_logger(os.path.join('./logger-'+now+'.log'))
device = 'cuda' if torch.cuda.is_available() else 'cpu'

best_acc = 0  # best test accuracy
start_epoch = 0  # start from epoch 0 or last checkpoint epoch

# Data
logger.info('==> Preparing data..')
transform_train = transforms.Compose([
    transforms.RandomCrop(32, padding=4),
    transforms.RandomHorizontalFlip(),
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

transform_test = transforms.Compose([
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

trainset = torchvision.datasets.CIFAR10(
    root='./data', train=True, download=True, transform=transform_train)
trainloader = torch.utils.data.DataLoader(
    trainset, batch_size=128, shuffle=True, num_workers=0)

testset = torchvision.datasets.CIFAR10(
    root='./data', train=False, download=True, transform=transform_test)
testloader = torch.utils.data.DataLoader(
    testset, batch_size=100, shuffle=False, num_workers=0)

classes = ('plane', 'car', 'bird', 'cat', 'deer',
           'dog', 'frog', 'horse', 'ship', 'truck')

# Model
logger.info('==> Building model..')
#net = VGG('VGG16')
net = VGG_vanilla('VGG19').cuda()
#net = ResNet56()
# net = PreActResNet18()
# net = GoogLeNet()
# net = DenseNet121()
# net = ResNeXt29_2x64d()
# net = MobileNet()
#net = MobileNetV2()
# net = DPN92()
# net = ShuffleNetG2()
# net = SENet18()
# net = ShuffleNetV2(1)
# net = EfficientNetB0()
# net = RegNetX_200MF()
# net = SimpleDLA()
net = net.to(device)
if device == 'cuda':
    net = torch.nn.DataParallel(net)
    cudnn.benchmark = True

if args.resume:
    # Load checkpoint.
    logger.info('==> Resuming from checkpoint..')
    assert os.path.isdir('checkpoint'), 'Error: no checkpoint directory found!'
    checkpoint = torch.load('./checkpoint/ckpt.pth')
    net.load_state_dict(checkpoint['net'])
    best_acc = checkpoint['acc']
    start_epoch = checkpoint['epoch']

criterion = nn.CrossEntropyLoss()
optimizer = optim.SGD(net.parameters(), lr=args.lr,
                      momentum=0.9, weight_decay=5e-4)
scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, T_max=200)


# Training
def train(epoch):
    logger.info('\nEpoch: %d' % epoch)
    net.train()
    train_loss = 0
    correct = 0
    total = 0
    for batch_idx, (inputs, targets) in enumerate(trainloader):
        inputs, targets = inputs.to(device), targets.to(device)
        optimizer.zero_grad()
        outputs = net(inputs)
        loss = criterion(outputs, targets)
        loss.backward()
        optimizer.step()

        train_loss += loss.item()
        _, predicted = outputs.max(1)
        total += targets.size(0)
        correct += predicted.eq(targets).sum().item()

        progress_bar(batch_idx, len(trainloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
                     % (train_loss/(batch_idx+1), 100.*correct/total, correct, total))


def test(epoch):
    global best_acc
    net.eval()
    test_loss = 0
    correct = 0
    total = 0
    with torch.no_grad():
        for batch_idx, (inputs, targets) in enumerate(testloader):
            inputs, targets = inputs.to(device), targets.to(device)
            outputs = net(inputs)
            loss = criterion(outputs, targets)

            test_loss += loss.item()
            _, predicted = outputs.max(1)
            total += targets.size(0)
            correct += predicted.eq(targets).sum().item()

            progress_bar(batch_idx, len(testloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
                         % (test_loss/(batch_idx+1), 100.*correct/total, correct, total))

    # Save checkpoint.
    acc = 100.*correct/total
    if acc > best_acc:
        logger.info('Saving..')
        state = {
            'net': net.state_dict(),
            'acc': acc,
            'epoch': epoch,
        }
        if not os.path.isdir('checkpoint'):
            os.mkdir('checkpoint')
        torch.save(state, './checkpoint/ckpt.pth')
        best_acc = acc

def get_layerwise(model):
    layerwise = []
    with torch.no_grad():
        for n,m in model.named_modules():
            if hasattr(m, 'mask'):
                mask = m.fixed_mask
                preserve = torch.sum(mask) / mask.numel()
                layerwise.append(float(preserve.cpu().numpy()))
    logger.info(layerwise)
    return layerwise

def get_prune_rate(model):
    layerwise = get_layerwise(model)
    input = torch.randn(1, 3, 32, 32).cuda()
    model_t = VGG_vanilla('VGG19').cuda()

    flops, params = profile(model_t, inputs=(input, ))
    #import pdb; pdb.set_trace()
    #layerwise=[0.5]*17
    model_t = VGG_vanilla('VGG19', layerwise=layerwise).cuda()

    flops_prune, params_prune = profile(model_t, inputs=(input, ))
    #import pdb; pdb.set_trace()
    return 1 - flops_prune/flops

def freeze_mask(model):
    for n,m in model.named_modules():
        if hasattr(m, 'mask'):
            m.mask.requires_grad_(False)


for epoch in range(start_epoch, start_epoch+200):
    train(epoch)
    #prune_rate = get_prune_rate(net)
    #if prune_rate > args.target_ratio:
    #    freeze_mask(net)
    logger.info('current prune_rate: {}'.format(prune_rate))
    test(epoch)
    logger.info('准确率',best_acc)
    scheduler.step()

layerwise = get_layerwise(net)
input = torch.randn(1, 3, 32, 32)
model = VGG('VGG19')

flops, params = profile(model, inputs=(input, ))


model = VGG('VGG19', layerwise=layerwise)

flops_prune, params_prune = profile(model, inputs=(input, ))
flops_ratio = flops_prune/flops
params_ratio = params_prune/params
flops, params = clever_format([flops, params], "%.3f")
flops_prune, params_prune = clever_format([flops_prune, params_prune], "%.3f")

logger.info('Before Pruning: Model FLOPS: {} Params: {}'.format(flops, params))
logger.info('After Pruning: Model FLOPS: {} Params: {}'.format(flops_prune, params_prune))
logger.info('FLOPS Compress Ratio: {} Params Compress Ratio: {}'.format(flops_ratio, params_ratio))
