# README: Digital Elevation Model (DEM) Processing and Export using Google Earth Engine (GEE)

## Project Overview

This project demonstrates the workflow for loading, visualizing, clipping, and exporting a Digital Elevation Model (DEM) from a user asset in Google Earth Engine (GEE). The DEM dataset represents elevation data for the Ranchi region in India, and the Area of Interest (AOI) is defined using a custom polygon.

The primary objectives include:
- Loading a DEM image from GEE Assets.
- Visualizing the DEM with custom parameters.
- Clipping the DEM to a user-defined AOI.
- Exporting the clipped DEM as a GeoTIFF to Google Drive.

---

## Code Description

### 1. Module Imports and Initialization

```python
import ee
import geemap
```

- **Earth Engine (ee)** is initialized to interact with GEE.
- **geemap** is used for interactive mapping in Python.

```python
try:
    ee.Initialize()
except Exception as e:
    ee.Authenticate()
    ee.Initialize()
```

- The script attempts to initialize GEE; if not authenticated, it prompts user authentication.

### 2. Loading the Image

```python
image = ee.Image("projects/ee-pallabkumardas/assets/ranchidem")
```

- Loads the DEM image from a GEE user asset path.
- **ranchidem** is a mosaiced GeoTIFF image derived from two individual scenes **P5_PAN_CS_N22_000_E085_000_30m** and **P5_PAN_CS_N23_000_E085_000_30m**, imaged by the PAN sensor of the CartoSat-1 satellite.

### 3. Visualization Parameters

Visualization parameters include:
- **Bands:** Band 'b1' (representing elevation values).
- **Min/Max:** Display value range for optimal contrast.
- **Gamma:** Adjusts image brightness.
- **Opacity:** Set to full visibility.

```python
image_vis_params = {
    'bands': ['b1'],
    'min': 165.15546478271483,
    'max': 635.8050302124024,
    'gamma': 1,
    'opacity': 1
}
```

### 4. Basic Image Information

Prints the selected band information, available band names, spatial projection details, and nominal spatial resolution to verify image metadata.

```python
print(image.select('b1'))
print(image.bandNames().getInfo())
print(image.select('b1').projection().getInfo())
print(image.select('b1').projection().nominalScale().getInfo())
```

### 5. Defining the Area of Interest (AOI)

A rectangular polygon defines the spatial extent of the clipping operation. Coordinates cover the Ranchi region in India.

```python
AOI = ee.Geometry.Polygon([
    [
        [85.26630201215673, 23.43198840049475],
        [85.26630201215673, 22.992770934206497],
        [85.7538203227036, 22.992770934206497],
        [85.7538203227036, 23.43198840049475],
        [85.26630201215673, 23.43198840049475]  # Polygon closure point
    ]
])
```

### 6. Creating an Interactive Map

Initializes a geemap interactive map clearly displaying both the full DEM and the clipped DEM.

```python
Map = geemap.Map(center=[23.212, 85.51], zoom=10)
Map.addLayer(image, image_vis_params, 'DEM Ranchi (Full)')
Map.addLayer(image.clip(AOI), image_vis_params, 'DEM Ranchi (Clipped)')
Map
```

### 7. Exporting the Image to Google Drive

Exports the clipped DEM image to your Google Drive account.

**Export Parameters:**
- **image:** Clipped DEM dataset.
- **description:** Name of the export task.
- **folder:** Destination folder in Google Drive.
- **fileNamePrefix:** Prefix for the exported file name.
- **region:** AOI boundary coordinates.
- **scale:** Spatial resolution (90 meters).
- **crs:** EPSG:4326 (WGS 84 Geographic Coordinate System).
- **maxPixels:** High pixel limit to handle large exports.
- **fileFormat:** GeoTIFF suitable for GIS applications.

```python
export_task = ee.batch.Export.image.toDrive(
    image=image.clip(AOI),
    description='ranchidem',
    folder='earthengine',
    fileNamePrefix='ranchidem',
    region=AOI,
    scale=90,
    crs='EPSG:4326',
    maxPixels=1e13,
    fileFormat='GeoTIFF'
)
export_task.start()
```

---

## Notes

- **Authentication:** Ensure you have authenticated and initialized the Earth Engine API correctly.
- **Asset Availability:** Confirm the existence and accessibility of the asset (`projects/ee-pallabkumardas/assets/ranchidem`).
- **Google Drive Permissions:** Verify Google Drive API access and ensure adequate storage space is available.
- **Interactive Map:** The geemap interactive map renders effectively in Jupyter Notebooks or Google Colab.
- **Export Monitoring:** Track export task progress in the Earth Engine "Tasks" tab. Upon completion, the GeoTIFF will be available in the specified Google Drive folder.

---

## Output

- **File:** `ranchidem.tif`
- **Format:** GeoTIFF
- **Spatial Extent:** Defined by AOI polygon.
- **Coordinate Reference System:** EPSG:4326
- **Resolution:** 90 meters per pixel.

---

## References

- [Google Earth Engine Python API Documentation](https://developers.google.com/earth-engine/guides/python_install)
- [Geemap Documentation](https://geemap.org/)
- [GeoTIFF Format Specification](https://gdal.org/drivers/raster/gtiff.html)

---

## Author

- **Prepared by:** Pallab Kumar Das
- **Date:** April 12, 2025
- **Affiliation:** Birla Institute of Technology, Mesra, Ranchi, Jharkhand-835215

---

*This README file guides users through DEM visualization, clipping, and export using Google Earth Engine and Python. For issues or queries, refer to the Earth Engine Developer Community or the Geemap GitHub repository.*