import csv  
import matplotlib.pyplot as plt  
import numpy as np  
from scipy.spatial import KDTree 
from importlib import import_module

def read_csv_points(filename):
    metadata_list = []  
    coordinates_list = [] 
    
    required_columns = ["Name", "Type", "Type_code", "WGS84_lon", "WGS84_lat","name_english"]
    
    with open(filename, 'r', encoding='UTF-8-sig') as csvfile:
        reader = csv.reader(csvfile)
        headers = None
        for row in reader:
            if row and row[0].startswith('#'):
                continue  
            elif row:  
                headers = [col.strip() for col in row]
                break
        
        if not headers:
            raise ValueError("The header row was not found in the CSV file")
        
       
        col_indices = {}
        try:
            for col in required_columns:
                col_indices[col] = headers.index(col)
        except ValueError as e:
            missing_col = str(e).split("'")[1]
            raise ValueError(f"The required columns are missing from the header row：{missing_col}") from e
        idx= 0
   
        for row in reader:
            if not row or row[0].startswith('#'):
                continue  
            
          
            if len(row) < max(col_indices.values()) + 1:
                continue
            
            try:
         
                idx = idx+1
                name = row[col_indices["Name"]].strip()
                type_ = row[col_indices["Type"]].strip()
                type_code = int(row[col_indices["Type_code"]])
                metadata_list.append((idx, name, type_, type_code))
                
       
                lon = float(row[col_indices["WGS84_lon"]])
                lat = float(row[col_indices["WGS84_lat"]])
                coordinates_list.append((lon, lat))
                
            except (ValueError, IndexError):
                continue  
    
    metadata_dtype = [
        ('idx', int),
        ('Name', 'U255'), 
        ('name_english', 'U255'),   
        ('Type', 'U255'),
        ('Type_code', int)
    ]
    


    metadata = np.array(metadata_list, dtype=metadata_dtype)
    coordinates = np.array(coordinates_list)
    
    return metadata, coordinates

 
def delaunay_triangulation(points):    
    node_ids = np.arange(len(points))   
    tree = KDTree(points)    
    nearest_neighbors_pairs = []    
    for i, node_id in enumerate(node_ids):  
        distances, nn_indices = tree.query(points[i], k=6)
        nn_pairs = [(node_id, nn_indices[j] ,distances[j]) for j in range(1, 6)]  
        nearest_neighbors_pairs.extend(nn_pairs)  
    

     
    nearest_neighbors_flat = [(node_id, nn_indices[j],distances[j]) for i, node_id in enumerate(node_ids) for j in range(1, 6)]  
    
 
    print ("Length",len(nearest_neighbors_flat))
    return nearest_neighbors_flat
    

  
def makedata():
    embedding = 'embedding_SougouNews.npz'
    model_name = "GeoDFnet"
    x = import_module('Models.' + model_name)
    config = x.Config(embedding)
    metadata, coordinates = read_csv_points(config.POIdata)
    edges = delaunay_triangulation(coordinates)

    with open(config.edge_path, 'w', newline='') as csvfile:  
        writer = csv.writer(csvfile)  
        writer.writerow(['Vertex1', 'Vertex2','len'])    
        for edge in edges:  
            writer.writerow(edge)  

    with open(config.data_path, 'w', encoding='UTF-8') as f:
        for item in metadata:
            line = f"{item['idx']}\t{item['Name']}\t{item['Type']}\t{item['Type_code']}\n"
            f.write(line)
        