import numpy as np
from typing import Dict

def compute_metrics(predictions: np.ndarray, targets: np.ndarray) -> Dict[str, float]:
    """Compute evaluation metrics.
    
    Args:
        predictions: Model predictions
        targets: Ground truth targets
        
    Returns:
        Dictionary of metric names and values
    """
    # Mean Squared Error
    mse = np.mean((predictions - targets) ** 2)
    
    # Root Mean Squared Error
    rmse = np.sqrt(mse)
    
    # Mean Absolute Error
    mae = np.mean(np.abs(predictions - targets))
    
    # R-squared
    ss_total = np.sum((targets - np.mean(targets)) ** 2)
    ss_residual = np.sum((targets - predictions) ** 2)
    r2 = 1 - (ss_residual / ss_total)
    
    return {
        'mse': float(mse),
        'rmse': float(rmse),
        'mae': float(mae),
        'r2': float(r2)
    }

def compute_feature_importance_metrics(
    numerical_importance: np.ndarray,
    categorical_importance: np.ndarray,
    numerical_names: list,
    categorical_names: list
) -> Dict[str, float]:
    """Compute metrics for feature importance analysis.
    
    Args:
        numerical_importance: Importance scores for numerical features
        categorical_importance: Importance scores for categorical features
        numerical_names: Names of numerical features
        categorical_names: Names of categorical features
        
    Returns:
        Dictionary of feature importance metrics
    """
    # Combine all feature importances
    all_importance = np.concatenate([numerical_importance, categorical_importance])
    all_names = numerical_names + categorical_names
    
    # Sort features by importance
    sorted_idx = np.argsort(all_importance)[::-1]
    sorted_importance = all_importance[sorted_idx]
    sorted_names = [all_names[i] for i in sorted_idx]
    
    # Compute cumulative importance
    cumulative_importance = np.cumsum(sorted_importance) / np.sum(sorted_importance)
    
    # Find number of features needed for 80% importance
    n_features_80 = np.argmax(cumulative_importance >= 0.8) + 1
    
    # Get top features
    top_features = sorted_names[:n_features_80]
    
    return {
        'n_features_80': int(n_features_80),
        'top_features': top_features,
        'feature_importance': dict(zip(sorted_names, sorted_importance.tolist()))
    } 