from dataclasses import dataclass
from typing import Optional
import yaml

@dataclass
class ModelConfig:
    """Configuration for ExcelFormer model."""
    embedding_dim: int = 64
    n_blocks: int = 4
    n_heads: int = 32
    hidden_dim: int = 256
    dropout: float = 0.1
    gamma: float = 1e-4

@dataclass
class TrainingConfig:
    """Configuration for model training."""
    batch_size: int = 32
    n_epochs: int = 100
    learning_rate: float = 1e-3
    weight_decay: float = 1e-5
    early_stopping_patience: int = 5
    train_ratio: float = 0.8
    val_ratio: float = 0.1
    random_state: int = 42
    use_wandb: bool = True
    model_save_path: Optional[str] = None

@dataclass
class DataConfig:
    """Configuration for data preprocessing."""
    n_quantiles: int = 1000
    target_column: str = 'ESG_Overall'

@dataclass
class Config:
    """Complete configuration for the project."""
    model: ModelConfig
    training: TrainingConfig
    data: DataConfig
    
    @classmethod
    def from_yaml(cls, path: str) -> 'Config':
        """Load configuration from YAML file.
        
        Args:
            path: Path to YAML file
            
        Returns:
            Config object
        """
        with open(path, 'r') as f:
            config_dict = yaml.safe_load(f)
        
        return cls(
            model=ModelConfig(**config_dict['model']),
            training=TrainingConfig(**config_dict['training']),
            data=DataConfig(**config_dict['data'])
        )
    
    def to_yaml(self, path: str) -> None:
        """Save configuration to YAML file.
        
        Args:
            path: Path to save YAML file
        """
        config_dict = {
            'model': self.model.__dict__,
            'training': self.training.__dict__,
            'data': self.data.__dict__
        }
        
        with open(path, 'w') as f:
            yaml.dump(config_dict, f, default_flow_style=False)

def get_default_config() -> Config:
    """Get default configuration.
    
    Returns:
        Default Config object
    """
    return Config(
        model=ModelConfig(),
        training=TrainingConfig(),
        data=DataConfig()
    ) 