package CP_Algorithms;

import java.util.ArrayList;
import java.lang.Math;
import java.util.Random;

import DatacenterSpec.Container;
import DatacenterSpec.PM;
import DatacenterSpec.VM;
import ExpConfig.Result;
import WOA_Config.Whale;
/**
 * implemented to simulate the IGA mechanism by Zhang2019 paper
 * @author abdulelah
 *
 */
public class IGA {
	private ArrayList<Container> containerSpecList;
	private ArrayList<VM> vmSpecList;
	private ArrayList<PM> pmSpecList;
	
	private int popSize;								//represents population of chromosome 
	private ArrayList<ArrayList<VM>> population;		//represents a list of chromosome 
	private ArrayList<ArrayList<PM>> popPMlist;			//represents a list of pms 
	
	private int cNum;
	private int vNum;
	private int pNum;
	private int NW;
	private int iterN;
	private double OLT = .8;								//default overloading threshold is set to 80%  
	
	private Result optResult;
	
	private double theta = 1.55;
	private int pCorssOver = 100;
	
	public IGA(int l, int v, int p, int NW, int iterN, ArrayList<Container> containerSpecList, ArrayList<VM> vmSpecList, ArrayList<PM> pmSpecList)
	{
		this.cNum = l;
		this.vNum = v;
		this.pNum = p;
		this.NW = NW;
		this.iterN = iterN;
		
		this.containerSpecList = new ArrayList<Container>(containerSpecList);
		this.vmSpecList = new ArrayList<VM>(vmSpecList);
		this.pmSpecList = new ArrayList<PM>(pmSpecList);
		
		this.population = new ArrayList<ArrayList<VM>>();
		this.popPMlist = new ArrayList<ArrayList<PM>>();
		this.popSize = this.NW;
		
		this.optResult = new Result("IGA");
	}
	public IGA(int l, int v, int p, int NW, int iterN, ArrayList<Container> containerSpecList, ArrayList<VM> vmSpecList, ArrayList<PM> pmSpecList, double OLT)
	{
		this(l, v, p, NW, iterN, containerSpecList, vmSpecList, pmSpecList);
		this.OLT = OLT;
	}
	
	public void startAlgorithm()
	{
		Random rand = new Random();
		this.popInit();		
		double power =0;
		
		for(int iter = 0; iter < this.iterN; iter++)
		{
			for(int i =0; i < this.popSize; i++)
			{
				int indexA = rand.nextInt(this.popSize);
				int indexB = rand.nextInt(this.popSize);

				ArrayList<VM> chromA = this.population.get(indexA);
				ArrayList<VM> chromB = this.population.get(indexB);
				ArrayList<VM> chromChild = crossOver(chromA, chromB);
				
				chromChild = this.correct(chromChild);
				
				if(this.containerSpecList.size() < 1)
					;
				else if(this.controlPara(iter, this.iterN) > rand.nextDouble())
					chromChild = this.tpemo(chromChild);
				else
					chromChild = this.tsemo(chromChild);

				chromChild = this.correct(chromChild);
				
				this.population.add(chromChild);
				ArrayList<PM> pmList = new ArrayList<PM>(this.assignPMstoChrom(chromChild));
				this.popPMlist.add(pmList);			
			}
			this.population = new ArrayList<ArrayList<VM>>(this.sortUpdatePopulation(this.population));			
		}
		Whale bestSolution = new Whale(this.containerSpecList.size());
		
		this.popPMlist.set(0, this.removePMUnused(this.popPMlist.get(0)));
		
		bestSolution.setRandomSolution(this.vNum, this.pNum);
		bestSolution.setContainerListForVmList(this.population.get(0));
		bestSolution.setVmListForPmList(this.popPMlist.get(0));

		this.optResult.setContNum(this.containerSpecList.size());
		this.optResult.setPower(bestSolution.getTotPowerConsumption());
		this.optResult.setVmNum(bestSolution.getVmTotalUniqueNumber());
		this.optResult.setPmNum(bestSolution.getPmTotalUniqueNumber());
		this.optResult.setSolNum(this.NW);
		this.optResult.setSearchSpace(this.iterN);
		this.optResult.setAvgUtil(bestSolution.getAvgUtilLevel());
		this.optResult.setPMsOLT(bestSolution.getTotOverLoadedPMs(this.OLT));
		this.optResult.setSearchSpace(this.iterN);
		/***/

		this.optResult.setAvgUtil(this.getAvgUtilLevel(popPMlist.get(0)));
		
		/*
		this.optResult.setPower(this.getPowerinChrom(this.popPMlist.get(0)));
		this.optResult.setVmNum(this.population.get(0).size());
		this.optResult.setPmNum(getUniquePmNumber(this.popPMlist.get(0)));
		this.optResult.setSolNum(this.NW);
		this.optResult.setSearchSpace(this.iterN);/**/
	}

	private double getAvgUtilLevel(ArrayList<PM> pmList)
	{
		ArrayList<Integer> pmUniqueIDs = new ArrayList<Integer>(); 
		double totUtil = 0;
		
		for(PM pm: pmList) {
			
			if(!pmUniqueIDs.contains(pm.getId())){
				pmUniqueIDs.add(pm.getId());
				totUtil  += pm.getUtilization();
				
				//System.out.println("pm id " + pm.getId() + ", pm util = " + pm.getUtilization());
			}
		}
		
		return Math.round(100 * totUtil/pmUniqueIDs.size());
	}
	
	private ArrayList<PM> removePMUnused(ArrayList<PM> pmList)
	{
		ArrayList<PM> tmpPmList = new  ArrayList<PM>();
		
		for(PM pm: pmList)
		{
			if(pm.getUtilization() > 0)
				tmpPmList.add(pm);
		}
		
		return tmpPmList;
	}

	private int findBestSolution(ArrayList<Whale> solutionMatrix) {
		int minIndex = 0;
		double powerConsumption = solutionMatrix.get(0).getTotPowerConsumption();
		
		for(int i=1; i < solutionMatrix.size(); i++) {
			double tmpPower = solutionMatrix.get(i).getTotPowerConsumption();
			
			if(tmpPower < powerConsumption) {
				minIndex = i;
				powerConsumption = tmpPower;
			}
			else if (tmpPower == powerConsumption) {
				if (solutionMatrix.get(i).getVmTotalUniqueNumber() < solutionMatrix.get(minIndex).getVmTotalUniqueNumber()) {
					minIndex = i;
				}
			}
		}
		return minIndex;
	}
	private ArrayList<ArrayList<VM>> sortUpdatePopulation(ArrayList<ArrayList<VM>> chromList)
	{
		boolean sorted = false;
		ArrayList<VM> tmpVmList = null;
		ArrayList<PM> tmpPmList = null;
		
	    while (!sorted) 
	    {
	        sorted = true;
	        for (int i = 0; i < chromList.size() - 1; i++) 
	        {
	        	if(this.getPowerinChrom(this.popPMlist.get(i)) > this.getPowerinChrom(this.popPMlist.get(i+1)) )
	        	{
	        		tmpVmList = new ArrayList<VM>(chromList.get(i));
	        		chromList.set(i, new ArrayList<VM>(chromList.get(i+1)));
	        		chromList.set(i+1, new ArrayList<VM>(tmpVmList));
	                sorted = false;
	                
	                //swapping pmLists
	                tmpPmList = new ArrayList<PM>(this.popPMlist.get(i));
	                this.popPMlist.set(i, new ArrayList<PM>(this.popPMlist.get(i+1)));
	                this.popPMlist.set(i+1, new ArrayList<PM>(tmpPmList));
	        	}
	        }
	    }
	    
	    ArrayList<ArrayList<VM>> updatedChromList = new ArrayList<ArrayList<VM>>();
	    ArrayList<ArrayList<PM>> updatedPmList = new ArrayList<ArrayList<PM>>();
	    
	    for(int i = 0; i < this.popSize; i++)
	    {
	    	//tmpVmList = this.correct(chromList.get(i));
	    	tmpVmList = chromList.get(i);
	    	updatedChromList.add(tmpVmList);
	    	updatedPmList.add(this.popPMlist.get(i));
	    	
	    	if(chromList.get(i).size() != tmpVmList.size())
	    	{
	    		System.out.println("[IGA.sortUpdatePopulation] : aborting !!!");
	    		System.exit(0);
	    	}
	    }
	    
	    this.popPMlist = new ArrayList<ArrayList<PM>>(updatedPmList);
	    return updatedChromList;
	}
	
	private double getPowerinChrom(ArrayList<PM> pmList)
	{
		double power = 0;
		for(PM pm: pmList)
		{
			power += pm.getPowerConsumption();
		}
		
		return Math.round(power);
	}
	
	private int getUniquePmNumber(ArrayList<PM> pmList)
	{
		int pmCounter = 0;
		for(PM pm: pmList)
		{
			if(pm.getUtilization() > 0)
				pmCounter++;
		}
		return pmCounter;		
	}
	
	private double getMinPower(ArrayList<ArrayList<PM>> chromList)
	{
		double power = this.getPowerinChrom(chromList.get(0));
		int powerIndex = 0;
		
		for(int i=1; i < chromList.size(); i++)
		{
			if(power > this.getPowerinChrom(chromList.get(i)))
			{
				power = this.getPowerinChrom(chromList.get(i));
				powerIndex = i;
			}
		}
		return power;
	}
	
	private double vmListAvgUtil(ArrayList<VM> chrom)
	{
		double util = 0;
		for(VM vm: chrom)
		{
			util = util + vm.getTotAllocatedMips()/vm.getMips();
		}
		
		return util/chrom.size();
	}
	
	public void test_ControlParm()
	{
		Random rand = new Random();
		
		for(int t=0; t < this.iterN; t++)
		System.out.println("this.controlPara = " + this.controlPara(t, this.iterN) + ", rand.nextDouble() = " + rand.nextDouble() + " ... exiting");
		System.exit(0);
	}
	
	public void test()
	{
		Random rand = new Random();
		int indexA = rand.nextInt(this.popSize);
		int indexB = rand.nextInt(this.popSize);
		
		System.out.println("indexA = " + indexA + " -- indexB = " + indexB);
		
		ArrayList<VM> chromA = this.population.get(indexA);
		ArrayList<VM> chromB = this.population.get(indexB);
		ArrayList<VM> chromChild = crossOver(chromA, chromB);
		
		System.out.println("printing chrom Child");
		this.printChrom(chromChild);

		chromChild = this.tpemo(chromChild);
		System.out.println("@@@ printing tpemo: chrom Child");
		this.printChrom(chromChild);

		chromChild = this.tsemo(chromChild);
		System.out.println("### printing tSemo: chrom Child");
		this.printChrom(chromChild);
		
		System.out.println("printing new & organized chrom Child");
		chromChild = this.setManyContOneVM(chromChild);
		this.printChrom(chromChild);
		
		
	}
	
	/**
	 * produces a new child chromosome based on mating two parents
	 * @param chromA The A parent
	 * @param chromB The B parent
	 * @return chromChild which is a list of cont->vm which is 1:1 
	 */
	private ArrayList<VM> crossOver(ArrayList<VM> chromA, ArrayList<VM> chromB)
	{
		ArrayList<VM> chromChild = new ArrayList<VM>();
		
		Random rand = new Random();
		int chromCtr = 0;
		int crossOver = rand.nextInt(this.cNum);
		
		if (crossOver == 0)
			crossOver++;
		else if (crossOver == this.cNum - 1)
			crossOver--;
		
		//System.out.println("crossOver = " + crossOver);
		
		boolean crossFlag = true;
		if((rand.nextInt() % 100) - 100 == 0)
			crossFlag = false;
		
		for(int vmIndex = 0; vmIndex < chromA.size(); vmIndex++)
		{
			if(chromCtr == crossOver && crossFlag)
				break;
						
			ArrayList<Container> containerList = chromA.get(vmIndex).getContainerList();
			
			for(int contIndex = 0; contIndex < containerList.size(); contIndex++)
			{
				if(chromCtr == crossOver && crossFlag)
					break;
				
				VM vmTmp = new VM(chromA.get(vmIndex).getId(), chromA.get(vmIndex).getRam(), chromA.get(vmIndex).getMips());
				vmTmp.addContainer(containerList.get(contIndex));
				chromChild.add(vmTmp);
				
				chromCtr++;
			}
		}

		if(crossFlag)
		{
			chromCtr = 0;
			for(int vmIndex = 0; vmIndex < chromB.size(); vmIndex++)
			{						
				ArrayList<Container> containerList = chromB.get(vmIndex).getContainerList();
				
				for(int contIndex = 0; contIndex < containerList.size(); contIndex++)
				{				
					VM vmTmp = new VM(chromB.get(vmIndex).getId(), chromB.get(vmIndex).getRam(), chromB.get(vmIndex).getMips());
					vmTmp.addContainer(containerList.get(contIndex));
	
					if(chromCtr >= crossOver)
					{
						chromChild.add(vmTmp);
					}
					
					chromCtr++;
				}
			}
		}
		
		return chromChild;
	}
	
	/**
	 * assign a list of pms to accommodate a list of vms in a chrom
	 * @param chrom a list of vms
	 * @return a list of pms which accomodate the list of vms
	 */
	private ArrayList<PM> assignPMstoChrom(ArrayList<VM> chrom)
	{
		Random rand = new Random();
		ArrayList<PM> tmpPMlist = new ArrayList<PM>(this.pmSpecList);
		for(VM vm: chrom)
		{			
			boolean allocateFlag = false;
			while(allocateFlag == false)
			{
				int pmIndex = rand.nextInt(tmpPMlist.size());
				PM pm = new PM(tmpPMlist.get(pmIndex).getId(), tmpPMlist.get(pmIndex).getRam(), tmpPMlist.get(pmIndex).getMips());
				pm.setVmList(tmpPMlist.get(pmIndex).getVmList());
				
				if(pm.canHost(vm))
				{
					pm.addVm(vm);
					tmpPMlist.set(pmIndex, pm);
					allocateFlag = true;
				}
			}
		}
		/*
		for(PM pm: tmpPMlist)
		{
			if(pm.getUtilization() > 0)
				System.out.println("pm id = " + pm.getId() + " -- # of vms: " + pm.getVmList().size());
		}*/
		
		return tmpPMlist;
	}
	
	private void printPmList(ArrayList<PM> pmList)
	{
		for(PM pm: pmList)
		{
			if(pm.getPowerConsumption() == 0)
				continue;
			
			System.out.print("pm.id: " + pm.getId() + " -- power: " + pm.getPowerConsumption() + " -- ");
		}
		System.out.println();
	}
	
	private ArrayList<VM> setManyContOneVM(ArrayList<VM> chrom)
	{
		ArrayList<VM> newChrom = new ArrayList<VM>();
		ArrayList<Integer> vmList = new ArrayList<Integer>();
		
		for(int vmIndex=0; vmIndex < chrom.size(); vmIndex++)
		{
			int vmId = chrom.get(vmIndex).getId();
			
			if(vmList.contains(vmId))
				continue;
			else
				vmList.add(vmId);
			
			ArrayList<Container> containerList = new ArrayList<Container>(chrom.get(vmIndex).getContainerList());
			for(int i=vmIndex+1; i < chrom.size(); i++)
			{
				if(vmId == chrom.get(i).getId())
				{
					ArrayList<Container> tmpContainerList = new ArrayList<Container>(chrom.get(i).getContainerList());
					containerList.addAll(tmpContainerList);
				}
			}
			VM vm = new VM(chrom.get(vmIndex).getId(), chrom.get(vmIndex).getRam(), chrom.get(vmIndex).getMips());
			vm.setContainerList(containerList);
			newChrom.add(vm);
		}
		return newChrom;
	}
	
	private ArrayList<VM> tpemo(ArrayList<VM> chrom)
	{
		chrom = this.setOneContOneVM(chrom);
		
		Random rand = new Random();
		int p1 = rand.nextInt(chrom.size());
		int p2 = rand.nextInt(chrom.size());
		
		//System.out.println("p1 = " + p1 + " -- p2 = " + p2);
		
		VM vm1 = chrom.get(p1);
		VM vm2 = chrom.get(p2);
		
		ArrayList<Container> tmpConList = new ArrayList<Container>(vm2.getContainerList());
		vm2.setContainerList(vm1.getContainerList());
		vm1.setContainerList(tmpConList);
		
		chrom.set(p1, vm2);
		chrom.set(p2, vm1);
		
		return chrom;
	}

	private ArrayList<VM> tsemo(ArrayList<VM> chrom)
	{
		chrom = this.setOneContOneVM(chrom);
		
		Random rand = new Random();
		
		int p1e = rand.nextInt( Math.abs(chrom.size() - 3 - 1) ) + 2;		
		int p1s = rand.nextInt(p1e);	
		

		int p2e = rand.nextInt( Math.abs(chrom.size() - p1e + 1) ) + p1e;
		if (p2e == chrom.size())
			p2e--;
		if(p2e == p1e)
			p2e++;
		//System.out.println("p1s = " + p1s + " -- p1e = " + p1e + " -- p2e = " + p2e);

		int vmIndex = p1s;
		ArrayList<Container> tmpConList = new ArrayList<Container>();
		ArrayList<VM> s1VMs = new ArrayList<VM>();
		ArrayList<VM> s2VMs = new ArrayList<VM>();
		
		//setting the first segment
		for(vmIndex = p1s; vmIndex <= p1e; vmIndex++)
		{
			VM vm = chrom.get(vmIndex);
			tmpConList.addAll(vm.getContainerList());			
			s1VMs.add(vm);
		}
		//setting the second segment
		for(vmIndex = p1e+1; vmIndex <= p2e; vmIndex++)
		{
			VM vm = chrom.get(vmIndex);
			tmpConList.addAll(vm.getContainerList());
			s2VMs.add(vm);
		}
		
		vmIndex = p1s;
		for(VM vm: s2VMs)
		{
			chrom.set(vmIndex, vm);
			vmIndex++;
		}

		for(VM vm: s1VMs)
		{
			chrom.set(vmIndex, vm);
			vmIndex++;
		}
		
		vmIndex = p1s;
		for(Container container: tmpConList)
		{
			VM vm = chrom.get(vmIndex);
			vm.setContainerList(new ArrayList<Container>());
			vm.addContainer(container);
			chrom.set(vmIndex, vm);
			vmIndex++;
		}
		
		return chrom;
	}
	
	private ArrayList<VM> setOneContOneVM(ArrayList<VM> chrom)
	{
		ArrayList<VM> newChrom = new ArrayList<VM>();
		for(VM vm: chrom)
		{
			for(Container cont: vm.getContainerList())
			{
				VM tmpVm = new VM(vm.getId(),vm.getRam(), vm.getMips());
				tmpVm.addContainer(cont);
				newChrom.add(tmpVm);
			}
		}
		return newChrom;
	}
	
	public void test_correct()
	{
		//test only on the first chromosome
		ArrayList<VM> chrom = this.population.get(0);
		
		VM vm_1 = chrom.get(0);
		VM vm_2 = chrom.get(1);
		
		for(Container container: vm_2.getContainerList())
		{
			vm_1.addContainer(container);	
		}
		//remove all containers
		vm_2.setContainerList(new ArrayList<Container>());
		
		chrom.set(0, vm_1);
		//chrom.set(1, vm_2);
		chrom.remove(1);

		this.population.set(0, chrom);
		System.out.println("*** printing pop after removing vm #2 without correcting");
		this.printPopulation();
		
		//test correct method
		chrom = this.correct(chrom);		
		this.population.set(0, chrom);
		System.out.println("### printing pop after removing vm #2 AFTER correcting");
		this.printPopulation();
	}
	
	/**
	 * to initialize the list of chromosomes which is container -> vms only
	 */
	private void popInit()
	{
		int vmIndex = 0;
		boolean contPlaceFlg;
		
		/**** printing ***/
		//System.out.println("IGA.popInit: this.containerSpecList.size() = " + this.containerSpecList.size());
		
		
		for(int index = 0; index < this.popSize; index++)
		{
			ArrayList<VM> chrom = new ArrayList<VM>();
			VM tmpVM = new VM(this.vmSpecList.get(vmIndex).getId(), this.vmSpecList.get(vmIndex).getRam(), this.vmSpecList.get(vmIndex).getMips());
			tmpVM.setContainerList(this.vmSpecList.get(vmIndex).getContainerList());;
			chrom.add(tmpVM);
			//chrom.add(this.vmSpecList.get(vmIndex));
			
			for(int i=0; i < this.cNum; i++)	//take container one by one, create a solution in a chromosome of containers placed to vms
			{
				Container cont = this.containerSpecList.get(i);
				contPlaceFlg = false;		//flag to indicate if the container has been placed
				
				for(int j=0; j < chrom.size(); j++)
				{
					if(chrom.get(j).canHost(cont)) 
					{
						tmpVM = new VM(chrom.get(j).getId(), chrom.get(j).getRam(), chrom.get(j).getMips());
						tmpVM.setContainerList(chrom.get(j).getContainerList());
						
						tmpVM.addContainer(cont);
						chrom.set(j, tmpVM);
						contPlaceFlg = true;
						break;
					}
				}
				if(contPlaceFlg == false)
				{
					vmIndex++;
					tmpVM = new VM(this.vmSpecList.get(vmIndex).getId(), this.vmSpecList.get(vmIndex).getRam(), this.vmSpecList.get(vmIndex).getMips());
					tmpVM.setContainerList(this.vmSpecList.get(vmIndex).getContainerList());;
					chrom.add(tmpVM);
					i--;					//the container has not been placed to a vm
				}
			}
			
			vmIndex = 0;
			this.population.add(chrom);
			
			ArrayList<PM> pmList = new ArrayList<PM>(this.assignPMstoChrom(chrom));
			//pmList = this.assignPMstoChrom(chrom);
			this.popPMlist.add(pmList);
			
			//this.printPmList(pmList);
			//System.out.println("\t\t ## power consumption = ##" + this.getPowerinChrom(pmList) );
		}
	}

	private double controlPara(double t, double maxI)
	{
		double p = 0;
		
		p = 2 * t/maxI;	//
		p = Math.pow(p - 1, 2);
		p = 1 - Math.cbrt(p);
		p = 1 - Math.cbrt(Math.pow( p , 2))/this.theta;
		return p;
	}
	
	/**
	 * check and correct a crom 
	 * @param chrom
	 */
	private ArrayList<VM> correct(ArrayList<VM> chrom)
	{
		chrom = this.setManyContOneVM(chrom);
		
		ArrayList<Container> containerList_toMigrate = new ArrayList<Container>();
		
		for(int vmIndex=0; vmIndex < chrom.size(); vmIndex++)
		{
			//the host vm is overloaded, remove containers and put them in a list
			if(chrom.get(vmIndex).isOverloaded())
			{
				//create a copy of the vm spec
				VM vm = new VM(chrom.get(vmIndex).getId(), chrom.get(vmIndex).getRam(), chrom.get(vmIndex).getMips());
				
				ArrayList<Container> containerListinVM = new ArrayList<Container>(chrom.get(vmIndex).getContainerList());
				for(int contIndex =0; contIndex < containerListinVM.size(); contIndex++)
				{
					if(vm.canHost(containerListinVM.get(contIndex)))
					{
						vm.addContainer(containerListinVM.get(contIndex));
					}
					else
					{
						containerList_toMigrate.add(containerListinVM.get(contIndex));
					}
				}
				//put back the vm after correcting it i.e. removing container(s) that causing overloading 
				chrom.set(vmIndex, vm);
			}
		}

		//allocate the removed containers to a suitable vm
		for(Container container: containerList_toMigrate)
		{
			boolean allocated = false;
			for(int vmIndex=0; vmIndex < chrom.size(); vmIndex++)
			{
				if(chrom.get(vmIndex).canHost(container))
				{
					VM vm = chrom.get(vmIndex);
					vm.addContainer(container);
					chrom.set(vmIndex, vm);
					
					allocated = true;
					break;
				}
			}
			
			if(allocated == false)		//if not allocated, select a new vm from vmSpec and allocate the container to it
			{
				VM tmpVM = getNewVM(chrom);
				VM vm = new VM(tmpVM.getId(), tmpVM.getRam(), tmpVM.getMips());
				vm.addContainer(container);
				
				chrom.add(vm);
			}
		}
		
		return chrom;
	}
	
	/**
	 * search for a new vm in vmSpecList that does not already host containers
	 * @return
	 */
	private VM getNewVM(ArrayList<VM> chrom)
	{
		boolean found;
		VM vm = null;
		for(int vmIndex=0; vmIndex < this.vmSpecList.size(); vmIndex++)
		{
			found = false;
			for(int i=0; i < chrom.size(); i++)
			{
				if(this.vmSpecList.get(vmIndex).getId() == chrom.get(i).getId())		//the vm already hosts container(s)
				{
					found = true;
					break;
				}
			}
			if(found == false)
			{
				vm = new VM(this.vmSpecList.get(vmIndex).getId(), this.vmSpecList.get(vmIndex).getRam(), this.vmSpecList.get(vmIndex).getMips());
				break;
			}
		}
		
		if(vm == null)
		{
			System.out.println("[IGA.getNewVM]: no more vms available, aborting the run");
			System.exit(0);
		}
		return vm;
	}
	
	/**
	 * to print one solution
	 * @param chrom
	 */	
	private void printChrom(ArrayList<VM> chrom)
	{
		for(VM vm: chrom)
		{
			ArrayList<Container> containerListinVM = vm.getContainerList();
			System.out.print("vm id " + vm.getId() + " : ");
			for(int i=0; i < containerListinVM.size(); i++)
				System.out.print( containerListinVM.get(i).getId() + " || ");

			System.out.println();
			//for(int i=0; i < containerListinVM.size(); i++) System.out.print( vm.getId() + " || ");
		}
		
		System.out.println();
	}
	
	private void printPopulation()
	{
		System.out.println("printing chromosomes: ");
		int counter = 0;
		
		for(ArrayList<VM> chrom: this.population)
		{
			System.out.println("print chrome # " + counter);
			this.printChrom(chrom);
			counter++;
		}
	}

	private void printPopulation(ArrayList<ArrayList<VM>> chromList)
	{
		System.out.println("printing chromosomes: ");
		int counter = 0;
		
		for(ArrayList<VM> chrom: chromList)
		{
			System.out.println("print chrome # " + counter);
			this.printChrom(chrom);
			counter++;
		}
	}

	public void printPopVM_PM()
	{
		for(PM pm: this.popPMlist.get(0))
		{
			if(pm.getVmList().size() == 0)
				continue;
			
			System.out.print("pm id: " + pm.getId() + " vms: ");
			
			for(VM vm: pm.getVmList())
				System.out.print(vm.getId() + " -- ");
			
			System.out.println("");
		}
		
		System.out.println("printing vms");
		
		for(VM vm: this.population.get(0))
		{
			if(vm.isOverloaded())
			{
				System.out.println("[IGA.printPopVM_PM]: vm id: " + vm.getId() + " --- is overloaded aborting");
				System.exit(0);
			}
			
			if(vm.getContainerList().size() == 0)
				continue;
			
			System.out.print("vm id: " + vm.getId() + " vms: ");
			
			for(Container cont: vm.getContainerList())
				System.out.print(cont.getId() + " -- ");
			
			System.out.println("");
		}
	}
	
	public Whale retSolution(ArrayList<VM> vmList, ArrayList<PM> pmList)
	{
		Whale w = new Whale(this.cNum);
		/*
		for(PM pm: pmList)
		{
			for(VM vm: pm.getVmList())
				w.addRecord(vm.getId(), pm.getId());
		}
		
		w.setContainerListForVmList(vmList);
		w.setVmListForPmList(pmList);
		/**/
		return w;
	}

	public void printOptimumBrief()
	{
		System.out.println("total power consumed by IGA: " + this.getPowerinChrom(this.popPMlist.get(0)) + ", number of unique VMs: " + this.population.get(0).size() + ", number of unique PMs: " + getUniquePmNumber(this.popPMlist.get(0)));
	}

	public Result getOptResult()
	{
		return this.optResult;
	}
}
