import os
import time

import cv2
import numpy as np
import glob
from natsort import natsorted

def main(info='', path='result.txt'):
    rmse_all, apd_all = calculate('image/container/', 'image/host/')
    result = f"\n{time.strftime('%y/%m/%d-%H:%M')} | RMSE: {np.mean(rmse_all):.4f} | APD: {np.mean(apd_all):.2f}% | {info}"
    with open(path, 'a') as f:
        f.write(result)

def calculate(folder_GT, folder_Pred):
    RMSE_all = []
    APD_all = []
    img_list = sorted(glob.glob(folder_GT + '/*.png'))
    img_list = natsorted(img_list)

    for i, img_path in enumerate(img_list):
        base_name = os.path.splitext(os.path.basename(img_path))[0]
        im_GT = cv2.imread(img_path) / 255.0
        im_Pred = cv2.imread(os.path.join(folder_Pred, base_name + '.png')) / 255.0

        # 计算 RMSE 和 APD
        rmse = calculate_rmse(im_GT, im_Pred)
        apd = calculate_apd(im_GT, im_Pred)

        print('{:3d} - {:25}. \tRMSE: {:.6f}, \tAPD: {:.6f}%'.format(i + 1, base_name, rmse, apd))

        RMSE_all.append(rmse)
        APD_all.append(apd)

    print('Average: RMSE: {:.6f}, APD: {:.6f}%   Image Size: {:d}'.format(
        np.mean(RMSE_all),
        np.mean(APD_all),
        im_GT.shape[1]))

    return RMSE_all, APD_all

def calculate_rmse(observed, predicted):
    """ (RMSE)"""
    rmse = np.sqrt(np.mean((observed - predicted) ** 2))
    return rmse

def calculate_apd(observed, predicted):
    """ (APD)"""
    apd = np.mean(np.abs((observed - predicted) / (observed + 1e-10))) * 100  # Add a small constant to avoid division by zero
    return apd

if __name__ == '__main__':
    main('Testing RMSE and APD calculations')
