#!/usr/local/bin/python3

'''
    USAGE: clam-diff-stats-to-json.py <clam-diff-files>
    Creates a json file with the stats of the clam-diff files.
    The json file has the following structure:

'''

import re
import json

regex_name = r"\b(\w+)\.(?:main|\w+)\.json\b"
regex_name = re.compile(regex_name)

regex_equals = r"(Number of equals\s+:)\s+(\d+)"
regex_equals = re.compile(regex_equals)

regex_regessions = r"(Number of regressions\s+:)\s+(\d+)"
regex_regessions = re.compile(regex_regessions)

regex_1_gt_than_2 = r"Number of 1 being more precise than 2:\s*(\d+)"
regex_1_gt_than_2 = re.compile(regex_1_gt_than_2)

regex_1_lt_than_2 = r"Number of 1 being less precise than 2:\s*(\d+)"
regex_1_lt_than_2 = re.compile(regex_1_lt_than_2)

regex_incomparable = r"(Number of incomparable\s+:)\s+(\d+)"
regex_incomparable = re.compile(regex_incomparable)

regex_no_reg_found = r"(No regressions found on )(\d+)( invariants)"
regex_no_reg_found = re.compile(regex_no_reg_found)


def extract_stats(comparisions):
    stats = {}
    for comparision in comparisions:
        try:
            match_name = re.search(regex_name, comparision)
            if match_name:
                file_name = match_name.group(1)
                if file_name == "exam":
                    file_name = "qsort-exam"
                if file_name not in stats:
                    stats[file_name] = {}
                    stats[file_name]["total"] = 0
                    stats[file_name]["equals"] = 0
                    stats[file_name]["regressions"] = 0
                    stats[file_name]["1_gt_than_2"] = 0
                    stats[file_name]["1_lt_than_2"] = 0
                    stats[file_name]["incomparable"] = 0
            else:
                raise Exception(
                    "ERROR: Could not find file name in first line")
            match_no_reg_found = re.search(regex_no_reg_found, comparision)
            if match_no_reg_found:
                stats[file_name]["total"] += int(match_no_reg_found.group(2))
                stats[file_name]["equals"] += int(match_no_reg_found.group(2))
                stats[file_name]["regressions"] += 0
                stats[file_name]["1_gt_than_2"] += 0
                stats[file_name]["1_lt_than_2"] += 0
                stats[file_name]["incomparable"] += 0
                continue
            # Get number of equals
            match_equals = re.search(regex_equals, comparision)
            if match_equals:
                stats[file_name]["equals"] += int(match_equals.group(2))
                stats[file_name]["total"] += int(match_equals.group(2))
            else:
                raise Exception(
                    "ERROR: Could not find number of equals in file: " + file_name + "")
            # Get number of regressions
            match_regessions = re.search(regex_regessions, comparision)
            if match_regessions:
                stats[file_name]["regressions"] += int(
                    match_regessions.group(2))
                stats[file_name]["total"] += int(match_regessions.group(2))
            else:
                raise Exception(
                    "ERROR: Could not find number of regressions in file: " + file_name + "")
            # Get number of 1 being more precise than 2
            match_1_gt_than_2 = re.search(regex_1_gt_than_2, comparision)
            if match_1_gt_than_2:
                stats[file_name]["1_gt_than_2"] += int(
                    match_1_gt_than_2.group(1))
            else:
                raise Exception(
                    "ERROR: Could not find number of 1 being more precise than 2 in file: " + file_name + "")
            # Get number of 1 being less precise than 2
            match_1_lt_than_2 = re.search(regex_1_lt_than_2, comparision)
            if match_1_lt_than_2:
                stats[file_name]["1_lt_than_2"] += int(
                    match_1_lt_than_2.group(1))
            else:
                raise Exception(
                    "ERROR: Could not find number of 1 being less precise than 2 in file: " + file_name + "")
            # Get number of incomparable
            match_incomparable = re.search(regex_incomparable, comparision)
            if match_incomparable:
                stats[file_name]["incomparable"] += int(
                    match_incomparable.group(2))
            else:
                raise Exception(
                    "ERROR: Could not find number of incomparable in file: " + file_name + "")
        except Exception as e:
            print("EXCEPITION RAISED: " + str(e))
            continue
    return stats


def analyze_stats(file_name, stats):
    file = open(file_name, "r")
    lines = file.read()

    # Split on line "Comparing precision of ..."
    comparisions = lines.split("Comparing precision of ")
    comparisions.remove(comparisions[0])

    file_name = file_name.replace(".txt", "")
    stats[file_name] = extract_stats(comparisions)


if __name__ == "__main__":
    import sys
    if len(sys.argv) < 3:
        print("USAGE: dump_table.py [clam-diff-files] <output-file>")
        sys.exit(1)

    stats = {}
    for file_name in sys.argv[1:-1]:
        analyze_stats(file_name, stats)
    json.dump(stats, open(sys.argv[-1], "w"), indent=4, sort_keys=True)
