# Install and load necessary packages
if (!require("survival")) install.packages("survival")
if (!require("readxl")) install.packages("readxl")
if (!require("rms")) install.packages("rms")    # Install rms package if needed
if (!require("regplot")) install.packages("regplot")  # Install regplot package

library(survival)
library(readxl)
library(rms)
library(regplot)

# Load dataset (ensure correct encoding if Chinese characters are present)
data <- read_excel("model 1.xlsx")

# Convert categorical variables
data$Group <- as.factor(data$Group)

# Build Cox proportional hazards model
cox_model <- coxph(Surv(Follow_up_time, AE) ~ Group + FEV1, data = data)

# Show model summary
cox_summary <- summary(cox_model)
print(cox_summary)

# Extract and save results to CSV
cox_results <- data.frame(
  Variable = rownames(cox_summary$coefficients),
  Coef = cox_summary$coefficients[, "coef"],
  HR = exp(cox_summary$coefficients[, "coef"]),
  Lower_CI = cox_summary$conf.int[, "lower .95"],
  Upper_CI = cox_summary$conf.int[, "upper .95"],
  SE = cox_summary$coefficients[, "se(coef)"],
  Z = cox_summary$coefficients[, "z"],
  P_value = cox_summary$coefficients[, "Pr(>|z|)"]
)

# Plot a nomogram using regplot, adjusting layout and font sizes
regplot(cox_model, 
        observation = data[4,],        # Use data from one patient
        interval = "confidence",       # Show confidence interval
        title = "Nomogram",            # Set plot title
        plots = c("violin", "boxes"),  # Draw violin and box plots
        clickable = TRUE,              # Make the plot interactive
        failtime = c(12, 24, 36),      # Follow-up time points: 1, 2, and 3 years
        cex.axis = 1.0,                # Axis label font size
        cex.main = 1.5,                # Main title font size
        cex.sub = 1.0,                 # Subtitle font size
        cex.lab = 1.2,                 # X/Y axis label font size
        xfrac = 0.75,                  # Width proportion adjustment
        yfrac = 0.85,                  # Height proportion adjustment
        width = 20,                    # Plot width (inches)
        height = 20,                   # Plot height (inches)
        margins = c(6, 6, 6, 6)        # Margins for spacing
)
