#!/usr/bin/env python
# coding: utf-8

# In[1]:


import numpy as np
import pandas as pd
from sklearn.model_selection import train_test_split
from sklearn.preprocessing import StandardScaler, LabelEncoder
from sklearn.metrics import classification_report, accuracy_score
from keras.models import Model
from keras.layers import Input, Dense, Softmax
from keras.utils import to_categorical
import keras.backend as K
import random

# Load and preprocess data (replace with actual dataset path or loading)
data = pd.read_csv('customer_behavior.csv')  # Change to your dataset

# Basic preprocessing
data.drop(columns=['city', 'timestamp'], errors='ignore', inplace=True)  # Remove unnecessary columns
for col in data.select_dtypes(include='object').columns:
    data[col] = LabelEncoder().fit_transform(data[col].astype(str))
X = data.drop(columns=['target'])  # Replace 'target' with actual target column name
y = data['target']
y = LabelEncoder().fit_transform(y)
y = to_categorical(y)

# Train-test split
X_train, X_test, y_train, y_test = train_test_split(X, y, test_size=0.2, random_state=42)
scaler = StandardScaler()
X_train = scaler.fit_transform(X_train)
X_test = scaler.transform(X_test)

# Deep Stacked Autoencoder construction
def build_autoencoder(input_dim, encoding_dims):
    input_layer = Input(shape=(input_dim,))
    x = input_layer
    for dim in encoding_dims:
        x = Dense(dim, activation='relu')(x)
    for dim in reversed(encoding_dims[:-1]):
        x = Dense(dim, activation='relu')(x)
    output_layer = Dense(input_dim, activation='sigmoid')(x)
    autoencoder = Model(input_layer, output_layer)
    encoder = Model(input_layer, x)
    return autoencoder, encoder

# Dragonfly Algorithm to optimize learning rate
def dragonfly_optimize(obj_func, bounds=(0.0001, 0.1), dim=1, pop_size=10, max_iter=20):
    lb, ub = bounds
    positions = np.random.uniform(lb, ub, (pop_size, dim))
    velocities = np.zeros((pop_size, dim))
    fitness = np.array([obj_func(p[0]) for p in positions])
    best_idx = np.argmin(fitness)
    best_pos = positions[best_idx]

    for t in range(max_iter):
        inertia = 0.9 - t * (0.9 - 0.4) / max_iter
        for i in range(pop_size):
            r1, r2 = np.random.rand(), np.random.rand()
            new_velocity = inertia * velocities[i] + r1 * (best_pos - positions[i])
            new_position = positions[i] + new_velocity
            new_position = np.clip(new_position, lb, ub)
            new_fitness = obj_func(new_position[0])
            if new_fitness < fitness[i]:
                positions[i] = new_position
                fitness[i] = new_fitness
        best_idx = np.argmin(fitness)
        best_pos = positions[best_idx]
    return best_pos[0]

# Objective function for learning rate optimization
def evaluate_model(lr):
    autoencoder, encoder = build_autoencoder(X_train.shape[1], [128, 64])
    autoencoder.compile(optimizer='adam', loss='mse')
    autoencoder.fit(X_train, X_train, epochs=20, batch_size=32, verbose=0)
    features = encoder.predict(X_train)
    classifier_input = Input(shape=(features.shape[1],))
    x = Dense(64, activation='relu')(classifier_input)
    x = Dense(y_train.shape[1], activation='softmax')(x)
    classifier = Model(classifier_input, x)
    classifier.compile(optimizer='adam', loss='categorical_crossentropy', metrics=['accuracy'])
    classifier.fit(features, y_train, epochs=10, batch_size=32, verbose=0)
    val_features = encoder.predict(X_test)
    val_preds = classifier.predict(val_features)
    val_loss = K.eval(K.mean(K.square(val_preds - y_test)))
    return val_loss

# Run Dragonfly Optimization
optimal_lr = dragonfly_optimize(evaluate_model)
print("Optimal Learning Rate:", optimal_lr)

# Final model training with optimized learning rate
autoencoder, encoder = build_autoencoder(X_train.shape[1], [128, 64])
autoencoder.compile(optimizer='adam', loss='mse')
autoencoder.fit(X_train, X_train, epochs=30, batch_size=32)

features_train = encoder.predict(X_train)
features_test = encoder.predict(X_test)

input_layer = Input(shape=(features_train.shape[1],))
x = Dense(64, activation='relu')(input_layer)
x = Dense(y_train.shape[1], activation='softmax')(x)
final_model = Model(input_layer, x)
final_model.compile(optimizer='adam', loss='categorical_crossentropy', metrics=['accuracy'])
final_model.fit(features_train, y_train, epochs=30, batch_size=32)

# Evaluation
preds = final_model.predict(features_test)
print(classification_report(np.argmax(y_test, axis=1), np.argmax(preds, axis=1)))
print("Accuracy:", accuracy_score(np.argmax(y_test, axis=1), np.argmax(preds, axis=1)))


# In[ ]:




