function [Best_pos, Best_fitness, Best_Path, Iter_curve] = IDSMPSO(popsize, dim, xmax, xmin, fobj, maxgen)
%% PSO Parameter Definition
% subsize = popsize/3;            % Sub-population size

Path = {};
xmin = xmin .* ones(1, dim);
xmax = xmax .* ones(1, dim);

v_min = -1; v_max = 1; % Velocity range

% Learning factors
c1 = 1.49445; c2 = 1.49445; c3 = 1.49445;   

% Dynamic weight parameters
wmax = 0.9; wmin = 0.4; 

% Levy flight parameters
beta = 3/2; 
alpha = 0.01; % Step size control factor
% Population division ratio
phi1 = 0.25; 
phi2 = 0.5;
phi3 = 0.75;
% Predefined arrays
X = zeros(popsize, dim);             % Particle positions
V = zeros(popsize, dim);             % Particle velocities

Pbest = zeros(popsize, dim);         % Personal best positions
fPbest = zeros(popsize, 1);          % Fitness of personal best positions

Gbest = zeros(1, dim);               % Global best position
fGbest = 0;                          % Fitness of global best position

% lBestFunreg = zeros(subsize, 1);    % Sub-population best fitness (unused)
% lBestSreg = zeros(subsize, dim);    % Sub-population best positions (unused)

fhistory = [];                       % Iteration history (initialized but unused)

f = zeros(popsize, 1);               % Current fitness of all particles


%% Particle Swarm Initialization
for i = 1:popsize
    % Initialize positions within [xmin, xmax]
    X(i, :) = xmin + (xmax - xmin) .* rand(1, dim); 
    % Initialize velocities within [v_min, v_max]
    V(i, :) = (v_max - v_min) * rands(1, dim) + v_min;  
    % Calculate initial fitness
    tempval = fobj(X(i, :)); 
    f(i) = tempval;
end

% Initialize global best (find particle with minimum fitness)
[fmin, ind] = min(f);
favg = mean(f);  % Average fitness of initial population
fGbest = fmin;
Gbest = X(ind, :);

% Initialize personal best (initial positions = personal best initially)
fPbest = f;
Pbest = X;

% Calculate fitness thresholds for population division
fit_min = min(f); fit_max = max(f);
f1 = fit_min;
f2 = phi1 * (fit_max - fit_min) + fit_min;
f3 = phi2 * (fit_max - fit_min) + fit_min;
f4 = phi3 * (fit_max - fit_min) + fit_min;
f5 = fit_max;  % Upper threshold (unused)

% Divide population into subgroups based on fitness
best_subgroup = (f1 <= f) & (f < f2);    % Top-performing subgroup
good_subgroup = (f2 <= f) & (f < f3);    % Mid-top subgroup
bad_subgroup = (f3 <= f) & (f < f4);     % Mid-worst subgroup
worst_subgroup = (f4 <= f);              % Worst-performing subgroup


%% Main Iteration Loop
for i = 1:maxgen
    R1 = rand;  % Random probability for local search trigger
    
    % Main PSO update (triggered with 90% probability)
    if R1 <= 0.9
        for j = 1:popsize
            % Calculate dynamic inertia weight (w) based on particle's fitness subgroup
            w = 0;
            if f(j) >= f4
                w = wmax;  % Worst subgroup: use maximum inertia weight
            elseif (f3 <= f(j)) && (f(j) < f4)
                % Bad subgroup: linearly decrease w from wmax to wmin
                w = (f4 - f(j)) * (wmax - wmin) / (f4 - f3) + wmin;
            elseif (f2 <= f(j)) && (f(j) < f3)
                % Good subgroup: linearly decrease w from wmax to wmin
                w = (f3 - f(j)) * (wmax - wmin) / (f3 - f2) + wmin;
            else
                % Best subgroup: linearly decrease w with iteration
                w = wmax - i * (wmax - wmin) / maxgen;
            end

            % Update logic for different subgroups
            if best_subgroup(j) == 1 
                % Top subgroup: Standard PSO + Levy flight (for exploration)
                V(j, :) = w * V(j, :) + c1 * rand * (Pbest(j, :) - X(j, :)) + c2 * rand * (Gbest - X(j, :));
                % Clip velocity to [v_min, v_max]
                V(j, find(V(j, :) > v_max)) = v_max;
                V(j, find(V(j, :) < v_min)) = v_min;
                % Update position
                X(j, :) = X(j, :) + V(j, :);
                % Clip position to [xmin, xmax]
                X(j, find(X(j, :) > xmax)) = xmax(find(X(j, :) > xmax));
                X(j, find(X(j, :) < xmin)) = xmin(find(X(j, :) < xmin));
                % Update fitness
                f(j, :) = fobj(X(j, :));

                % Perform Levy flight (enhance exploration)
                Levy_step = levy(dim, beta);
                X1 = X(j, :) + alpha * Levy_step;
                % Boundary check for Levy flight position
                for k = 1:dim
                    if X1(k) > xmax(k)
                        X1(k) = xmax(k);
                    end
                    if X1(k) < xmin(k)
                        X1(k) = xmin(k);
                    end
                end
                % Evaluate fitness of Levy flight position
                f1 = fobj(X1);
               
                % Greedy selection: keep better position
                if f1 < f(j, :)
                    X(j, :) = X1;
                end

            elseif good_subgroup(j) == 1
                % Mid-top subgroup: Standard PSO (balance exploration/exploitation)
                V(j, :) = w * V(j, :) + c1 * rand * (Pbest(j, :) - X(j, :)) + c2 * rand * (Gbest - X(j, :));
                % Clip velocity
                V(j, find(V(j, :) > v_max)) = v_max;
                V(j, find(V(j, :) < v_min)) = v_min;
                % Update position
                X(j, :) = X(j, :) + V(j, :);
                % Clip position
                X(j, find(X(j, :) > xmax)) = xmax(find(X(j, :) > xmax));
                X(j, find(X(j, :) < xmin)) = xmin(find(X(j, :) < xmin));
                % Update fitness
                f(j, :) = fobj(X(j, :));

            elseif bad_subgroup(j) == 1
                % Mid-worst subgroup: PSO with adaptive learning factors
                V(j, :) = w * V(j, :) + 2 * cos(pi * i / (2 * maxgen)) * rand * (Pbest(j, :) - X(j, :)) + 2 * sin(pi * i / (2 * maxgen)) * rand * (Gbest - X(j, :));
                % Clip velocity
                V(j, find(V(j, :) > v_max)) = v_max;
                V(j, find(V(j, :) < v_min)) = v_min;
                % Update position
                X(j, :) = X(j, :) + V(j, :);
                % Clip position
                X(j, find(X(j, :) > xmax)) = xmax(find(X(j, :) > xmax));
                X(j, find(X(j, :) < xmin)) = xmin(find(X(j, :) < xmin));
                % Update fitness
                f(j, :) = fobj(X(j, :));

            else
                % Worst subgroup: PSO + mixed personal best + Gaussian mutation
                % Randomly select personal bests to form "pmix" (enhance diversity)
                numb = unidrnd(popsize, [1, dim]);
                for ii = 1:dim
                    pmix(ii) = Pbest(numb(ii), ii);
                end
                
                % Velocity update (add pmix term for diversity)
                V(j, :) = w * V(j, :) + c1 * rand * (Pbest(j, :) - X(j, :)) + c2 * rand * (Gbest - X(j, :)) + c3 * rand * (pmix - X(j, :));
                % Clip velocity
                V(j, find(V(j, :) > v_max)) = v_max;
                V(j, find(V(j, :) < v_min)) = v_min;
                % Update position
                X(j, :) = X(j, :) + V(j, :);
                % Clip position
                X(j, find(X(j, :) > xmax)) = xmax(find(X(j, :) > xmax));
                X(j, find(X(j, :) < xmin)) = xmin(find(X(j, :) < xmin));
                % Update fitness
                f(j, :) = fobj(X(j, :));
                
                % Gaussian mutation (trigger probability decreases with iteration)
                if rand > (1/2 * (1 + atan(i / maxgen) * 4 / pi))
                    X1 = X(j, :) .* (1 + randn(1, dim));  % Gaussian perturbation
                    % Boundary check for mutated position
                    X1(find(X1 > xmax)) = xmax(find(X1 > xmax));
                    X1(find(X1 < xmin)) = xmin(find(X1 < xmin));
                    % Evaluate fitness of mutated position
                    fNEW = fobj(X1);
                    % Greedy selection: keep better position
                    if fNEW < f(j, :)
                        X(j, :) = X1;
                        f(j, :) = fNEW;
                    end
                end
            end
        end
    
    % Local search (triggered with 10% probability: Quasi-Newton method)
    else
        % Configure fminunc (quasi-Newton) for local refinement
        options = optimoptions('fminunc', 'Algorithm', 'quasi-newton', 'Display', 'off');
        HH = fminunc(fobj, Gbest, options);  % Refine global best
        HH = BoundaryCheck(HH, xmax, xmin, dim);  % Ensure refined position is within bounds
        % Update global best if refined position is better
        if fobj(HH) < fobj(Gbest)
            Gbest = HH;
            Best_fitness = fobj(HH);
        end
    end

    % Re-divide population into subgroups (update fitness thresholds)
    fit_min = min(f); fit_max = max(f);
    f1 = fit_min;
    f2 = phi1 * (fit_max - fit_min) + fit_min;
    f3 = phi2 * (fit_max - fit_min) + fit_min;
    f4 = phi3 * (fit_max - fit_min) + fit_min;
    f5 = fit_max;
    best_subgroup = (f1 <= f) & (f < f2);
    good_subgroup = (f2 <= f) & (f < f3);
    bad_subgroup = (f3 <= f) & (f < f4);
    worst_subgroup = (f4 <= f);

    % Update personal best and global best
    for j = 1:popsize
        % Update personal best: keep if current position is better
        if f(j) < fPbest(j)
            Pbest(j, :) = X(j, :);
            fPbest(j) = f(j);
        end
        
        % Update global best: keep if current position is better
        if f(j) < fGbest
            Gbest = X(j, :);
            fGbest = f(j);
        end
    end

    % Record iteration curve (global best fitness per iteration)
    Iter_curve(i) = fGbest;
    % Update output variables
    Best_pos = Gbest;
    Best_fitness = fGbest;
end

% Find the optimal path (from the particle with minimum final fitness)
for i = 1:popsize
    [f(i), Path{i}] = fobj(X(i, :));  % Re-evaluate to get path for each particle
end
[~, idx] = sort(f);  % Sort particles by fitness (ascending)
Best_Path = Path{idx(1)};  % Select path of the best particle (minimum fitness)
end