function [fitness, path, collisionFlag] = calFitness(pos, startPos, goalPos, X, Y, Z, n)
% Extract path points
x = pos(1:n);
y = pos(n+1:2*n);
z = pos(2*n+1:end);

% Construct complete path sequence
x_seq = [startPos(1), x, goalPos(1)];
y_seq = [startPos(2), y, goalPos(2)];
z_seq = [startPos(3), z, goalPos(3)];

% Generate smooth path using cubic spline interpolation
k = length(x_seq);
i_seq = linspace(0,1,k);
I_seq = linspace(0,1,200);  % Increase number of interpolation points to improve collision detection accuracy
X_seq = spline(i_seq, x_seq, I_seq);
Y_seq = spline(i_seq, y_seq, I_seq);
Z_seq = spline(i_seq, z_seq, I_seq);
path = [X_seq', Y_seq', Z_seq'];

% Collision detection (core correction)
collisionFlag = 0;
safetyMargin = 0.5;  % Safety distance (drone radius + redundancy)
for i = 1:size(path,1)
    x_coord = path(i,1);
    y_coord = path(i,2);
    
    % Check if coordinates are within terrain range
    if x_coord < min(X(:)) || x_coord > max(X(:)) || ...
       y_coord < min(Y(:)) || y_coord > max(Y(:))
        collisionFlag = 1;  %жΪײ when outside terrain range
        break;
    end
    
    % Terrain elevation interpolation (using linear interpolation for stability)
    z_terrain = interp2(X, Y, Z, x_coord, y_coord, 'linear');
    
    % Check for collision (incorporating safety distance)
    if z_terrain - path(i,3) > safetyMargin
        collisionFlag = 1;
        break;
    end
end

% Calculate path length (fitness)
dx = diff(X_seq);
dy = diff(Y_seq);
dz = diff(Z_seq);
fitness = sum(sqrt(dx.^2 + dy.^2 + dz.^2));

% Collision penalty
if collisionFlag == 1
    fitness = 1000 * fitness;  % Amplify fitness for colliding paths
end
end
    