function [X, Y, Z] = defMap(mapRange)
% Generate more complex terrain to increase collision possibilities

% Mountain peak parameter settings
N = 15;  % Increase the number of mountain peaks
peaksInfo = repmat(struct('center', [], 'range', [], 'height', []), N, 1);

% Generate mountain peak features (increase randomness in height and range)
for i = 1:N
    % More uniform distribution of peak centers
    peaksInfo(i).center = [mapRange(1)*(rand*0.9+0.05), mapRange(2)*(rand*0.9+0.05)];
    % Peak height
    peaksInfo(i).height = mapRange(3)*(rand*0.8+0.2);
    % Larger peak range
    peaksInfo(i).range = mapRange*0.15*(rand*0.8+0.2);
end

% Generate valleys and complex terrain
peakData = zeros(mapRange(1), mapRange(2));
for x = 1:mapRange(1)
    for y = 1:mapRange(2)
        sumVal = 0;
        for k = 1:N
            h_i = peaksInfo(k).height;
            x_i = peaksInfo(k).center(1);
            y_i = peaksInfo(k).center(2);
            x_si = peaksInfo(k).range(1);
            y_si = peaksInfo(k).range(2);
            % Gaussian function to generate mountain peak shape
            sumVal = sumVal + h_i * exp(-((x-x_i)/x_si)^2 - ((y-y_i)/y_si)^2);
        end
        % Add random terrain undulations
        sumVal = sumVal + (rand*20 - 10);  % Add 10m random fluctuations
        peakData(x, y) = max(0, sumVal);  % Ensure altitude is not negative
    end
end

% Generate interpolation grid (improve terrain resolution)
[xMesh, yMesh] = meshgrid(1:mapRange(1), 1:mapRange(2));
xVec = xMesh(:);
yVec = yMesh(:);
zVec = peakData(:);

% Generate high-density terrain grid for collision detection
[X, Y] = meshgrid(linspace(1, mapRange(1), 200), linspace(1, mapRange(2), 200));
Z = griddata(xVec, yVec, zVec, X, Y, 'cubic');  % Cubic interpolation for more realistic terrain

% Fill possible NaN values
Z(isnan(Z)) = 0;
end
