clear, clc, close all;
rng(3);  % Fix random seed to ensure experimental reproducibility

%% 1. Basic Parameter Configuration (No Modifications)
startPos = [1, 1, 1];         % Start point coordinates
goalPos = [180, 180, 160];    % Goal point coordinates
mapRange = [200, 200, 300];   % Terrain map range [X-range, Y-range, Z-range]

% 3D Terrain Loading and Generation
if exist('MAP.mat', 'file')
    load('MAP.mat');
    fprintf('Terrain data loaded: MAP.mat\n');
else
    [X, Y, Z] = defMap(mapRange);
    save('MAP.mat', 'X', 'Y', 'Z');
    fprintf('Terrain data generated and saved: MAP.mat\n');
end

% Algorithm Parameters
pop = 30;                    % Population size
maxIter = 100;               % Maximum number of iterations
n = 5;                        % Number of path nodes
dim = 3 * n;                  % Variable dimension (x1,y1,z1,...,xn,yn,zn)
lb = ones(1, dim);            % Lower bounds of variables
ub = 199 * ones(1, dim);      % Upper bounds of variables
search_space = [lb; ub];      % Search space for GWO algorithm
fobj = @(x) calFitness(x, startPos, goalPos, X, Y, Z, n);  % Fitness function

% Algorithm List and Names (7 algorithms total)
algorithms = {
    @(pop,dim,fobj,ss,iter) GWO(pop,dim,fobj,ss,iter), ...
    @(pop,iter,lb,ub,dim,fobj) GOOSE(pop,iter,lb,ub,dim,fobj), ...
    @(pop,iter,lb,ub,dim,fobj) EO(pop,iter,lb,ub,dim,fobj), ...
    @(pop,iter,lb,ub,dim,fobj) KOA(pop,iter,lb,ub,dim,fobj), ...
    @(pop,dim,ub,lb,fobj,iter) IDMPSO(pop,dim,ub,lb,fobj,iter), ...
    @(pop,iter,dim,fobj) HPSOALS(pop,iter,dim,fobj), ...
    @(pop,dim,ub,lb,fobj,iter) IDSPSO(pop,dim,ub,lb,fobj,iter)
};
algNames = {'GWO', 'GOOSE', 'EO', 'KOA', 'IDMPSO', 'HPSOALS', 'IDSMPSO'};
numAlgs = length(algorithms);  % Should equal 7


%% 2. Collision Statistics Configuration (No Modifications)
totalExperiments = 30;  % Number of experiments per algorithm
collisionStats = zeros(numAlgs, 3);  % [Collision count, Collision rate, Avg fitness]
experimentLogs = cell(numAlgs, totalExperiments);  % Logs for each experiment

% Fix: Properly initialize struct array (ensure each element is a scalar struct)
algAllResults = cell(numAlgs, 1);
for algIdx = 1:numAlgs
    % Create and assign scalar struct
    algResult = struct();
    algResult.fitness = zeros(1, totalExperiments);  % Fitness values per experiment
    algResult.collisionFlag = zeros(1, totalExperiments);  % Collision status (1=collision)
    algResult.path = cell(1, totalExperiments);  % Optimal path per experiment
    algResult.iterCurve = cell(1, totalExperiments);  % Iteration curve per experiment
    % Store scalar struct in cell array
    algAllResults{algIdx} = algResult;
end


%% 3. Run Algorithms and Count Collisions (No Modifications)
fprintf('\nStarting %d algorithms (%d experiments per algorithm)...\n', numAlgs, totalExperiments);
for algIdx = 1:numAlgs
    currentAlgName = algNames{algIdx};
    currentAlgFunc = algorithms{algIdx};
    fprintf('\nTesting algorithm: %s (%d/%d)\n', currentAlgName, algIdx, numAlgs);
    
    collCount = 0;  % Count of collision cases
    fitnessList = [];  % List of fitness values for statistics
    rng(3);  % Fix random seed for consistency
    
    for expIdx = 1:totalExperiments
        % Call corresponding algorithm (adapt to different parameter orders)
        switch currentAlgName
            case 'GWO'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, dim, fobj, search_space, maxIter);
            case 'GOOSE'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, maxIter, lb, ub, dim, fobj);
            case 'EO'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, maxIter, lb, ub, dim, fobj);
            case 'KOA'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, maxIter, lb, ub, dim, fobj);
            case 'IDMPSO'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, dim, ub, lb, fobj, maxIter);
            case 'HPSOALS'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, maxIter, dim, fobj);
            case 'IDSMPSO'
                [Best_pos, Best_fitness, Best_Path, iterCurve] = currentAlgFunc(pop, dim, ub, lb, fobj, maxIter);
        end
        
        % Determine collision status (1 = collision, 0 = no collision)
        [~, ~, collisionFlag] = calFitness(Best_pos, startPos, goalPos, X, Y, Z, n);
        
        % Store current experiment results (including iteration curve)
        algAllResults{algIdx}.fitness(expIdx) = Best_fitness;
        algAllResults{algIdx}.collisionFlag(expIdx) = collisionFlag;
        algAllResults{algIdx}.path{expIdx} = Best_Path;
        algAllResults{algIdx}.iterCurve{expIdx} = iterCurve;  % Save iteration curve
        
        % Update statistics
        fitnessList = [fitnessList, Best_fitness];
        if collisionFlag == 1
            collCount = collCount + 1;
            logStr = sprintf('%s_Experiment%d: Collision | Fitness=%.2f', currentAlgName, expIdx, Best_fitness);
        else
            logStr = sprintf('%s_Experiment%d: No Collision | Fitness=%.2f', currentAlgName, expIdx, Best_fitness);
        end
        experimentLogs{algIdx, expIdx} = logStr;
        fprintf('  %s\n', logStr);
    end
    
    % Calculate statistical results
    collRate = (collCount / totalExperiments) * 100;  % Collision rate (%)
    avgFitness = mean(fitnessList);  % Average fitness
    collisionStats(algIdx, :) = [collCount, collRate, avgFitness];
end


%% 4. Output Statistical Results (No Modifications)
fprintf('\n===================== Statistical Results =====================\n');
statTable = table(algNames', collisionStats(:,1), collisionStats(:,2), collisionStats(:,3), ...
    'VariableNames', {'Algorithm_Name', 'Collision_Count', 'Collision_Rate(%)', 'Average_Fitness'});
disp(statTable);
save('Algorithm_Stats.mat', 'collisionStats', 'statTable', 'experimentLogs', 'algAllResults');


%% 5. Select Optimal Paths and Explicit Iteration Curves (Key Fix: Add 7 Iteration Curve Vars)
% Define 7 explicit path variables + 7 explicit iteration curve variables (1:1 with algorithms)
% Path variables
Best_Path1 = [];  % Optimal path for GWO
Best_Path2 = [];  % Optimal path for GOOSE
Best_Path3 = [];  % Optimal path for EO
Best_Path4 = [];  % Optimal path for KOA
Best_Path5 = [];  % Optimal path for IDMPSO
Best_Path6 = [];  % Optimal path for HPSOALS
Best_Path7 = [];  % Optimal path for IDSMPSO
% Iteration curve variables (New: Store iteration process of optimal paths)
Iter_curve1 = [];  % Optimal iteration curve for GWO
Iter_curve2 = [];  % Optimal iteration curve for GOOSE
Iter_curve3 = [];  % Optimal iteration curve for EO
Iter_curve4 = [];  % Optimal iteration curve for KOA
Iter_curve5 = [];  % Optimal iteration curve for IDMPSO
Iter_curve6 = [];  % Optimal iteration curve for HPSOALS
Iter_curve7 = [];  % Optimal iteration curve for IDSMPSO

fprintf('\n===================== Select Optimal Paths & Iteration Curves =====================\n');
for algIdx = 1:numAlgs
    currentAlgName = algNames{algIdx};
    results = algAllResults{algIdx};
    
    % Priority: Select paths with NO collisions (consistent with original logic)
    validIndices = find(results.collisionFlag == 0);
    if ~isempty(validIndices)
        % Choose path with minimum fitness from collision-free paths
        [~, bestIdx] = min(results.fitness(validIndices));
        bestExpIdx = validIndices(bestIdx);
        selectionType = 'Collision-free & optimal fitness';
    else
        % If all paths collide: choose path with minimum fitness
        [~, bestExpIdx] = min(results.fitness);
        selectionType = 'All paths collide; select optimal fitness';
    end
    
    % Assign values: Optimal path + corresponding iteration curve (Key Fix)
    switch algIdx
        case 1  % GWO
            Best_Path1 = results.path{bestExpIdx};
            Iter_curve1 = results.iterCurve{bestExpIdx};  % Assign optimal iteration curve
        case 2  % GOOSE
            Best_Path2 = results.path{bestExpIdx};
            Iter_curve2 = results.iterCurve{bestExpIdx};
        case 3  % EO
            Best_Path3 = results.path{bestExpIdx};
            Iter_curve3 = results.iterCurve{bestExpIdx};
        case 4  % KOA
            Best_Path4 = results.path{bestExpIdx};
            Iter_curve4 = results.iterCurve{bestExpIdx};
        case 5  % IDMPSO
            Best_Path5 = results.path{bestExpIdx};
            Iter_curve5 = results.iterCurve{bestExpIdx};
        case 6  % HPSOALS
            Best_Path6 = results.path{bestExpIdx};
            Iter_curve6 = results.iterCurve{bestExpIdx};
        case 7  % IDSMPSO
            Best_Path7 = results.path{bestExpIdx};
            Iter_curve7 = results.iterCurve{bestExpIdx};
    end
    
    % Print selection results (add iteration curve info)
    curveLen = length(results.iterCurve{bestExpIdx});
    fprintf('%s: Selected Experiment%d results (%s) | Fitness=%.2f | Iteration curve length=%d (should equal maxIter=%d)\n', ...
        currentAlgName, bestExpIdx, selectionType, results.fitness(bestExpIdx), curveLen, maxIter);
end


%% 6. Validate Optimal Paths & Iteration Curves (Key Fix: Add Curve Validation)
fprintf('\n===================== Validation of Optimal Paths & Iteration Curves =====================\n');
% Path validation (retain original logic)
pathVariables = {Best_Path1, Best_Path2, Best_Path3, Best_Path4, Best_Path5, Best_Path6, Best_Path7};
% Iteration curve validation (New: 1:1 correspondence with path variables)
curveVariables = {Iter_curve1, Iter_curve2, Iter_curve3, Iter_curve4, Iter_curve5, Iter_curve6, Iter_curve7};
validPathCount = 0;  % Count of valid paths
validCurveCount = 0;  % Count of valid iteration curves

for i = 1:numAlgs
    currentAlg = algNames{i};
    path = pathVariables{i};
    curve = curveVariables{i};
    fprintf('\n【%s】Validation Results:\n', currentAlg);
    
    % 1. Path validity check
    if isempty(path)
        fprintf('  ❌ Path: Empty!\n');
    elseif size(path, 2) ~= 3
        fprintf('  ❌ Path: Format error (should be N×3 coordinate matrix), current dimension: %d×%d\n', size(path,1), size(path,2));
    else
        fprintf('  ✅ Path: Valid (%d×3 coordinate matrix)\n', size(path,1), size(path,2));
        validPathCount = validPathCount + 1;
    end
    
    % 2. Iteration curve validity check (New)
    if isempty(curve)
        fprintf('  ❌ Iteration Curve: Empty!\n');
    elseif length(curve) ~= maxIter
        fprintf('  ❌ Iteration Curve: Length error (should equal maxIter=%d), current length: %d\n', maxIter, length(curve));
    elseif ~isnumeric(curve)
        fprintf('  ❌ Iteration Curve: Data type error (should be numeric array), current type: %s\n', class(curve));
    else
        fprintf('  ✅ Iteration Curve: Valid (length=%d, numeric array)\n', length(curve));
        validCurveCount = validCurveCount + 1;
    end
end

% Overall validation summary
fprintf('\n===================== Overall Validation Summary =====================\n');
fprintf('Path Validation: %d/7 valid\n', validPathCount);
fprintf('Iteration Curve Validation: %d/7 valid\n', validCurveCount);
if validPathCount == 7 && validCurveCount == 7
    fprintf('✅ All optimal paths and iteration curves are valid for subsequent plotting and analysis!\n');
    % Optional: Save explicit variables to .mat for separate use later
    save('Best_Results_Explicit.mat', ...
         'Best_Path1', 'Best_Path2', 'Best_Path3', 'Best_Path4', 'Best_Path5', 'Best_Path6', 'Best_Path7', ...
         'Iter_curve1', 'Iter_curve2', 'Iter_curve3', 'Iter_curve4', 'Iter_curve5', 'Iter_curve6', 'Iter_curve7', ...
         'algNames', 'maxIter');
    fprintf('✅ Explicit variables saved to: Best_Results_Explicit.mat\n');
else
    warning('⚠️ Invalid paths or iteration curves exist! Check algorithm output or selection logic.');
end


%% 7. Result Visualization (No Modifications; Reference补充代码 below for curve plotting)
% 7.1 3D Path Comparison Plot (pass parameters in required format)
if validPathCount == 7
    plotFigure(startPos, goalPos, X, Y, Z, ...
               Best_Path1, Best_Path2, Best_Path3, ...
               Best_Path4, Best_Path5, Best_Path6, Best_Path7);
    saveas(gcf, '3D_Optimal_Path_Comparison.png');
    fprintf('\n✅ 3D optimal path plot saved (includes 7 algorithms)\n');
else
    fprintf('\n❌ 3D path plot not generated (fewer than 7 valid paths)\n');
end

% 7.2 Collision Rate Bar Chart (No Modifications)
figure('Position', [200, 200, 800, 500]);
bar(1:numAlgs, collisionStats(:,2), 'FaceColor', [0.2, 0.6, 0.8]);
xlabel('Algorithm Name', 'FontSize', 11);
ylabel('Collision Rate (%)', 'FontSize', 11);
title('Collision Rate Comparison of 7 Algorithms (30 Experiments)', 'FontSize', 13);
set(gca, 'XTick', 1:numAlgs, 'XTickLabel', algNames, 'XTickLabelRotation', 45);
for i = 1:numAlgs
    text(i, collisionStats(i,2)+0.5, sprintf('(%d/%d)', collisionStats(i,1), totalExperiments), ...
        'HorizontalAlignment', 'center', 'FontSize', 9);
end
grid on;
saveas(gcf, 'Collision_Rate_Comparison.png');

% 7.3 Optimal Iteration Convergence Curve
figure
semilogy(Iter_curve1,'--','Color','k','LineWidth',1.5); hold on
semilogy(Iter_curve7,'--','Color','c','LineWidth',1.5); hold on
semilogy(Iter_curve3