clear, clc, close all;
rng(2);  % Fix random seed for reproducibility

%% Use CEC2022 Test Suite (Modify test suite as needed)
Func_name = 'F11';  % Name of the test function (F2 excluded)
funNum = 11;        % Corresponding number of the test function
dim = 10;           % Dimension of the test function
% Get function bounds, dimension, and objective function handle
[lb, ub, dim, fobj] = Get_Functions_cec2022(Func_name, dim);  

%% Algorithm Parameters
pop = 30;           % Population size
maxIter = 1000;     % Maximum number of iterations
k = 30;             % Number of independent runs
vmax = 1 * ones(1, dim);    % Maximum velocity
vmin = -1 * ones(1, dim);   % Minimum velocity
% Alternative velocity bounds (commented out):
% vmax = 0.2 * (ub - lb) .* ones(1, dim);
% vmin = 0.1 * ones(1, dim);

xmin = lb .* ones(1, dim);  % Lower bound of position
xmax = ub .* ones(1, dim);  % Upper bound of position

% Search space (fixed to [-100, 100] for all dimensions)
search_space = [-100 * ones(1, dim); 100 * ones(1, dim)];  

% Alternative global bounds (for consistency with some algorithms)
Lb = -100;
Ub = 100;

% Handle for CEC2017 function (commented out, unused in current logic)
% fhd = str2func('cec17_func');

% Initialize result structure
result = {};  % Store all run results
% List of intelligent optimization algorithms to test (add/remove as needed)
name_list = {'GWO', 'GO', 'EO', 'KOA', 'MPSO', 'EAPSO', 'HPSOALS', 'DMPSORH', 'IDSMPSO'};  

len = length(name_list);
result.name = name_list;                  % Load algorithm names into result
result.data = cell(1, len);               % Initialize cell array for iteration curves
for i = 1:len 
    % Each element stores k runs × maxIter iterations of fitness data
    result.data{i} = zeros(k, maxIter);
end

% Initialize initial population (for algorithms requiring pre-initialized positions)
X = lb + (ub - lb) .* rand(pop, dim);


%% Run All Algorithms
for i = 1:k  % Repeat for k independent runs
    fprintf('Running independent run %d/%d...\n', i, k);

    % Run each algorithm and record iteration curves
    % Note: Algorithm-function mappings are corrected for consistency (e.g., GWO calls EO in original, kept as-is)
    % GWO (Note: Original code calls EO; retain to match original logic)
    [~, eo_best_fit, Iter_curve1] = GWO(pop, dim, fobj, search_space, maxIter);
    % GO (GOOSE algorithm)
    [~, go_best_fit, Iter_curve2] = GOOSE(pop, maxIter, Lb, Ub, dim, fobj);
    % EO (Note: Original code calls GWO; retain to match original logic)
    [~, gwo_best_fit, Iter_curve3] = EO(pop, maxIter, Lb, Ub, dim, fobj);
    % KOA
    [~, koa_best_fit, Iter_curve4] = KOA(pop, maxIter, Lb, Ub, dim, fobj);
    % CLPSO (commented out; uncomment to enable)
    % [~, CLPSO_fitness, Iter_curve5] = CLPSO(pop, dim, fobj, search_space, maxIter);
    % MPSO
    [~, MPSO_fitness, Iter_curve5] = MPSO(pop, dim, maxIter, funNum, lb, ub);
    % EAPSO
    [~, EAPSO_fitness, Iter_curve6] = EAPSO(funNum, dim, lb, ub, pop, maxIter, X);
    % HPSOALS
    [~, HPSOALS_fitness, Iter_curve7] = HPSOALS(pop, maxIter, dim, fobj);
    % IDMPSO
    [~, IDMPSO_fitness, Iter_curve8] = IDMPSO(pop, maxIter, xmax, xmin, dim, fobj);
    % IDSPSO
    [~, IPSO_fitness, Iter_curve9] = IDSPSO(pop, maxIter, xmax, xmin, dim, fobj);

    % Store iteration curves for the current run
    result.data{1}(i, :) = Iter_curve1;  % GWO
    result.data{2}(i, :) = Iter_curve2;  % GO (GOOSE)
    result.data{3}(i, :) = Iter_curve3;  % EO
    result.data{4}(i, :) = Iter_curve4;  % KOA
    result.data{5}(i, :) = Iter_curve5;  % MPSO
    result.data{6}(i, :) = Iter_curve6;  % EAPSO
    result.data{7}(i, :) = Iter_curve7;  % HPSOALS
    result.data{8}(i, :) = Iter_curve8;  % DMPSORH (IDMPSO)
    result.data{9}(i, :) = Iter_curve9;  % IDSMPSO
end


%% Calculate Statistical Metrics
% Select algorithm index 'num' to compare with other algorithms (for Wilcoxon/Friedman tests)
num = 8;  % Index of the baseline algorithm (adjust as needed)

% Generate RGB color matrix (customize colors as needed; rows ≥ number of algorithms)
Colors = hsv(len);  % Generate hue-saturation-value color map
for i = 1:len
    RGBData(i, :) = Colors(i, :);
    % Lighten colors (add 10% white) for better plot visibility
    RGBData(i, :) = 0.9 * RGBData(i, :) + 0.1 * [1 1 1];
end

% Compute statistical metrics (calls external function Statis_Data)
[Best_val, Worst_val, Mean_val, Median_val, Std_val, Wilcoxon_RankSum_Test, ...
 Wilcoxon_SignedRank_Test, Friedamn_val] = Statis_Data(result, Func_name, num, RGBData, dim);


%% Print Statistical Results
disp('List of tested algorithms:');
disp(name_list);
disp('------------------------------');
disp('Best value:'); 
Best_val_str = sprintf('%.2f  ', Best_val); 
disp(Best_val_str);
disp('------------------------------');
disp('Worst value:'); 
Worst_val_str = sprintf('%.2f  ', Worst_val);
disp(Worst_val_str);
disp('------------------------------');
disp('Mean value:'); 
Mean_val_str = sprintf('%.2f  ', Mean_val);
disp(Mean_val_str);
disp('------------------------------');
disp('Median value:'); 
Median_val_str = sprintf('%.2f  ', Median_val);
disp(Median_val_str);
disp('------------------------------');
disp('Standard deviation:'); 
Std_val_str = sprintf('%.2f  ', Std_val);
disp(Std_val_str);
disp('------------------------------');
disp('Wilcoxon Rank-Sum Test results:'); 
disp(Wilcoxon_RankSum_Test);
disp('------------------------------');
disp('Wilcoxon Signed-Rank Test results:'); 
disp(Wilcoxon_SignedRank_Test);
disp('------------------------------');
disp('Friedman test statistic:'); 
Friedamn_val_str = sprintf('%.2f  ', Friedamn_val); 
disp(Friedamn_val_str);
disp('------------------------------');