﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Bloomberglp.Blpapi;

using CDSHeatMap.ConfigurationManager;
using CDSHeatMap.Interface;
using CDSHeatMap.Model;
using CDSHeatMap.Shared;
using System.Diagnostics;
using System.Windows.Threading;
using CDSHeatMap.View;
using Bloomberglp.AppPortalApi.Desktop;
using System.Threading;

namespace CDSHeatMap.BloombergAPIManager
{
    public class MarketDataProvider : IMarketDataProvider
    {
        private Session session = null;
        private Session sessionAsync = null;
        public event Action<IMarketDataSubscription> SubsriptionDataUpdate;

        public bool initialized = false;

        Dispatcher mainThreadDispatcher;

        private ApiLogger _logger = new ApiLogger();

        private class ApiLogger : Logging.Callback
        {
            #region Callback Members

            public void OnMessage(long threadId, TraceLevel level, Datetime dateTime, string loggerName, string message)
            {
                try
                {
                    Trace.WriteLine(string.Format("### BlpApiMessage ###   {3}    Level: {0}    LoggerName: {1}    Message: {2}",
                                    level, loggerName, message, dateTime), "INFO");
                }
                catch (Exception)
                {
                }

            }

            #endregion
        }


        public bool isInitialized()
        {
            return initialized;
        }

        SessionOptions sessionOptions;

        public MarketDataProvider()
        {
            Logging.RegisterCallback(_logger, TraceLevel.Info);

            Trace.WriteLine("MarketDataProvider is called.", "TRACE");

            mainThreadDispatcher = Dispatcher.CurrentDispatcher;

            try
            {
                sessionOptions = new SessionOptions
                {
                    ServerHost = "localhost",
                    ServerPort = 8194,
                    ClientMode = SessionOptions.ClientModeType.DAPI,
                    AutoRestartOnDisconnection = true,
                    ConnectTimeout = 60000
                };

                #region Create_SessionAsync
                sessionAsync = createSession(true);
                
                if (sessionAsync == null)
                {
                    Trace.WriteLine("Failed to create Bloomberg session to get market data (2010)!!!", "INFO");

                    string errMessage = "bbcomm.exe is not running on your machine. In order to use any of the APPS, bbcomm.exe should be started.";

                    throw new BlpApiDataException(errMessage);
                }

                if (!sessionAsync.OpenService("//blp/mktdata"))
                {
                    Trace.WriteLine("Failed to open Bloomberg Api Market Data Service with session async. (2012)", "INFO");

                    string errMessage = "Failed to open market data service for Bloomberg session!!!";

                    throw new BlpApiDataException(errMessage);
                }
                #endregion

                #region Create_Session

                session = createSession();

                if (session == null)
                {
                    Trace.WriteLine("Failed to start session with Bloomberg Api Services. (2014)", "INFO");

                    string errMessage = "bbcomm.exe is not running on your machine. In order to use any of the APPS, bbcomm.exe should be started.";

                    throw new BlpApiDataException(errMessage);
                }

                if (!session.OpenService("//blp/refdata"))
                {
                    Trace.WriteLine("Failed to open Bloomberg Api Reference Data Service (2016)", "INFO");

                    string errMessage = "Failed to open reference data service for Bloomberg session!!!";

                    throw new BlpApiDataException(errMessage);
                }

                if (!session.OpenService("//blp/instruments"))
                {
                    Trace.WriteLine("Failed to open Bloomberg Api Instruments Data Service (2018)", "INFO");

                    string errMessage = "Failed to open instrument service for Bloomberg session!!!";

                    throw new BlpApiDataException(errMessage);
                }
                #endregion

                Trace.WriteLine("CDSHeatMap application is connected to host=localhost and port=8194.", "INFO");

                initialized = true;

                BLPSessionManager.getInstance();

                Trace.WriteLine("MarketDataProvider is completed.", "TRACE");
            }
            catch (Exception e)
            {
                initialized = false;
                Trace.WriteLine("CDSHeatMap application cannot create session (2020): " + e.Message.ToString(), "INFO");
                string stackTrace = "";
                if(e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (2020): " + stackTrace, "INFO");

                string errMessage = e.Message.ToString();

                if (e.GetType() == typeof(BlpApiDataException))
                {
                    mainThreadDispatcher.BeginInvoke((Action)delegate
                    {
                        try
                        {
                            ModalDialog.Instance.Title = "API Error";
                            ModalDialog.Instance.ClearMaxHeight();
                            ModalDialog.Instance.ClearMaxWidth();
                            ModalDialog.Instance.ClearMinHeight();
                            ModalDialog.Instance.ClearMinWidth();

                            string errMainMessage = "CDSHeatmap could not be started...";

                            ModalDialog.Instance.ShowDialog(new ErrorMessageDialog(errMainMessage + "\n\n" + e.Message.ToString(), ErrorMsgLevel.BLPContact));

                            mainThreadDispatcher.BeginInvokeShutdown(DispatcherPriority.Normal);
                        }
                        catch (Exception eDispatcher)
                        {
                            Trace.WriteLine("Exception while mainThreadDispatcher process : " + eDispatcher.Message.ToString(), "INFO");
                            string stackTraceDispatcher = "";
                            if (eDispatcher.StackTrace != null)
                                stackTraceDispatcher = eDispatcher.StackTrace.ToString();

                            Trace.WriteLine("Exception Stacktrace: " + stackTraceDispatcher, "INFO");
                        }
                    });
                     
                }
                else
                {
                    mainThreadDispatcher.BeginInvoke((Action)delegate
                    {
                        try
                        {
                            ModalDialog.Instance.Title = "API Error";
                            ModalDialog.Instance.Width = 500;
                            ModalDialog.Instance.Height = 400;

                            string errMainMessage = "CDSHeatmap could not be started...";

                            ModalDialog.Instance.ShowDialog(new DisplayExceptionView(errMainMessage + "\n\n" + e.Message.ToString(), stackTrace));
                        }
                        catch (Exception eDispatcher)
                        {
                            Trace.WriteLine("Exception while mainThreadDispatcher process : " + eDispatcher.Message.ToString(), "INFO");
                            string stackTraceDispatcher = "";
                            if (eDispatcher.StackTrace != null)
                                stackTraceDispatcher = eDispatcher.StackTrace.ToString();

                            Trace.WriteLine("Exception Stacktrace: " + stackTraceDispatcher, "INFO");
                        }
                    });
                }
            }
        }

        public Session createSession(bool isAsync=false)
        {
            //Trace.WriteLine("createSession is called.", "TRACE");

            var retries = 3;

            while (true)
            {
                Session _session = null;
                if(isAsync)
                    _session = new Session(sessionOptions, ProcessEventCallBack);
                else
                    _session = new Session(sessionOptions);

                if (_session.Start())
                {
                    return _session;
                }
                else
                    _session = null;

                if (retries == 0)
                {
                    if (isAsync)
                        Trace.WriteLine("Failed to start async session with Bloomberg Api Services.", "INFO");
                    else
                        Trace.WriteLine("Failed to start session with Bloomberg Api Services.", "INFO");
                    //throw new BlpApiDataException("Failed to start session with Bloomberg Api Services.");
                    //throw new BlpApiDataException(mainThreadDispatcher, "Failed to start session with Bloomberg API Services.");

                    return null;
                }
                retries--;
            }
        }

        private void ProcessSessionEvent(Session _session)
        {
            bool done = false;
            while (!done)
            {
                Event eventObj = _session.NextEvent();
                if (eventObj.Type == Event.EventType.PARTIAL_RESPONSE)
                {
                    ProcessEventCallBack(eventObj);
                }
                else if (eventObj.Type == Event.EventType.RESPONSE)
                {
                    ProcessEventCallBack(eventObj);
                    done = true;
                }
                else
                {
                    foreach (Message msg in eventObj)
                        if (eventObj.Type == Event.EventType.SESSION_STATUS)
                        {
                            if (msg.MessageType.Equals(BBCommMessageType.SESSIONTERMINATED))
                            {
                                done = true;
                            }
                        }
                }
            }

        }

        private void ProcessEventCallBack(Event eventObject, Session session = null)
        {
            try
            {
                foreach (var msg in eventObject)
                {
                    switch (msg.MessageType.ToString())
                    {
                        case BBCommMessageType.SESSIONSTARTED:
                            Trace.WriteLine("SESSIONSTARTED!!!", "INFO");
                            break;
                        case BBCommMessageType.SESSIONSTARTUPFAILURE:
                            Trace.WriteLine("SESSIONSTARTUPFAILURE!!!", "INFO");
                            break;
                        case BBCommMessageType.SESSIONTERMINATED:
                            Trace.WriteLine("SESSIONTERMINATED!!!", "INFO");
                            break;
                        case BBCommMessageType.REFERENCEDATARESPONSE:
                            OnProcessReferenceDataResponse(msg);
                            break;
                        case BBCommMessageType.HISTORICALDATARESPONSE:
                            OnProcessHistoricalDataResponse(msg);
                            break;
                        case BBCommMessageType.MARKETDATAEVENTS:
                            OnProcessMarketDataResponse(msg);
                            break;
                        case BBCommMessageType.INSTRUMENTLISTRESPONSE:
                            OnProcessInstrumentListResponse(msg);
                            break;
                        default:
                            //Trace.WriteLine("Unexpected message in ProcessEventCallBack: " + msg.MessageType.ToString(), "INFO");
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                Trace.WriteLine(string.Format("Exception during processing Bloomberg Data API Services Event: {0}", ex.Message.ToString()), "ERROR");
                //throw new BlpApiDataException(string.Format("Exception during processing Bloomberg Data API Services Event: {0}", ex));

                mainThreadDispatcher.BeginInvoke(((Action)delegate
                {
                    string errMessage = "Unexpected Error while retrieving data from Bloomberg...";
                    try
                    {
                        string stackTrace = "";
                        if (ex.StackTrace != null)
                            stackTrace = ex.StackTrace.ToString();

                        Trace.WriteLine("Exception Stacktrace (2050): " + stackTrace, "INFO");

                        var popup = new DisplayExceptionView(errMessage + "\n" + ex.Message.ToString(), stackTrace);
                        PopupWindow.Instance.Title = "API Error";
                        PopupWindow.Instance.ShowWindow(popup);
                        PopupWindow.Instance.Width = 500;//Aysegul Albayrak,popup size
                        PopupWindow.Instance.Height = 600;
                    }
                    catch (Exception e)
                    {
                        Trace.WriteLine("Exception while displaying Bloomberg API Error (2051): " + errMessage + "\n" + e.Message, "ERROR");

                        string stackTrace = "";
                        if (e.StackTrace != null)
                            stackTrace = e.StackTrace.ToString();

                        Trace.WriteLine("Exception Stacktrace (2051): " + stackTrace, "INFO");
                    }
                }), null);
            }
        }

        private void OnProcessMarketDataResponse(Message msg)
        {
            try
            {
                var apiSubscription = (IMarketDataSubscription)msg.CorrelationID.Object;

                var element = msg.AsElement;

                if (apiSubscription.DataFields == null)
                    return;

                if (apiSubscription.DataFields.Count <= 0)
                    return;


                foreach (var dataField in apiSubscription.DataFields.Where(dataField => element.HasElement(dataField.Name)))
                {
                    if (element.GetElement(dataField.Name).NumValues > 0)
                        dataField.Value = element.GetElement(dataField.Name).GetValue();
                    else
                        dataField.Value = null;
                }

                if (SubsriptionDataUpdate != null)
                    SubsriptionDataUpdate(apiSubscription);
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while procession Bloomberg Market Data (2052): " + e.Message + "\n" + msg.ToString(), "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (2052): " + stackTrace, "INFO");
            }
        }

        private void OnProcessReferenceDataResponse(Message msg)
        {
            //Trace.WriteLine("OnProcessReferenceDataResponse is called.", "TRACE");

            var asyncResult = (MarketDataResult)msg.CorrelationID.Object;

            if (msg.HasElement(BBCommMessageType.RESPONSEERROR) ||
                msg.HasElement(BBCommMessageType.REQUESTFAILURE))
            {
                mainThreadDispatcher.BeginInvoke(((Action)delegate
                {
                    try
                    {
                        string errMessage = string.Format("Failed to make request to API Services: {0}",
                            msg.HasElement(BBCommMessageType.RESPONSEERROR) ? msg.GetElement(BBCommMessageType.RESPONSEERROR) : msg.GetElement(BBCommMessageType.REQUESTFAILURE));

                        Trace.WriteLine(errMessage + "\n", "ERROR");

                        var popup = new NotificationView(errMessage);
                        PopupWindow.Instance.Title = "API Error";
                        PopupWindow.Instance.ShowWindow(popup);
                        PopupWindow.Instance.Width = 500;//Aysegul Albayrak,popup size
                        PopupWindow.Instance.Height = 400;
                    }
                    catch (Exception e)
                    {
                        Trace.WriteLine("Exception while displaying Bloomberg API Error (2053): " + e.Message, "ERROR");
                        string stackTrace = "";
                        if (e.StackTrace != null)
                            stackTrace = e.StackTrace.ToString();

                        Trace.WriteLine("Exception Stacktrace (2053): " + stackTrace, "INFO");
                    }
                }), null);

                return;
                //throw new BlpApiDataException(mainThreadDispatcher, string.Format("Failed to process reference data response to Bloomberg Data API Services: {0}", msg.GetElement(BBCommMessageType.RESPONSEERROR)));
            }

            try
            {
                if (!msg.HasElement(Names.SECURITYDATA))
                {
                    Trace.WriteLine("OnProcessReferenceDataResponse, message doesn't include securityData: " + msg.ToString() + "\n", "INFO");
                    return;
                }

                var securities = msg.GetElement(Names.SECURITYDATA);

                for (int i = 0; i < securities.NumValues; ++i)
                {
                    var security = securities.GetValueAsElement(i);
                    string ticker = security.GetElementAsString(Names.SECURITY);

                    if (security.HasElement(Names.SECURITYERROR))
                    {
                        Trace.WriteLine(string.Format("Failed to process reference data response for Security {1} Exception: {0}", security.GetElement(Names.SECURITYERROR), ticker), "ERROR");

                        continue;
                    }

                    var values = security.GetElement(Names.FIELDDATA);

                    if (values != null)
                    {
                        if (values.HasElement(DataFieldNames.INDX_MEMBERS))
                        {
                            Element fields = values.GetElement(DataFieldNames.INDX_MEMBERS);

                            if (fields.NumValues > 0)
                            {
                                List<IndexMember> IndexMemberList = new List<IndexMember>();
                                int numElements = fields.NumValues;
                                for (int j = 0; j < numElements; ++j)
                                {
                                    Element field = fields.GetValueAsElement(j);
                                    Element f1 = field.GetElement("Member Ticker and Exchange Code");
                                    Element f4 = field.Elements.Last();
                                    IndexMemberList.Add(new IndexMember(f1.GetValueAsString(), f4.GetValueAsString()));
                                }

                                if (asyncResult.DataFields[0].Name == DataFieldNames.INDX_MEMBERS && IndexMemberList.Count > 0)
                                    asyncResult.DataFields[0].Value = IndexMemberList;
                            }
                        }
                        else
                        {
                            List<IDataField> fieldList;
                            if (asyncResult.TickerDataFields.TryGetValue(ticker, out fieldList))
                            {
                                foreach (var field in fieldList)
                                {
                                    var element = values.GetElement(field.Name);

                                    if (element != null)
                                    {
                                        try
                                        {
                                            if (element.NumValues > 0)
                                                field.Value = element.GetValue();
                                            else
                                                field.Value = null;
                                        }
                                        catch (Exception ex)
                                        {
                                            Trace.WriteLine(string.Format("Error: (2054). Failed to retrieve reference data for Security {1} Field {2} Exception: {0}", ex.Message, ticker, field.Name), "ERROR");

                                            string stackTrace = "";
                                            if (ex.StackTrace != null)
                                                stackTrace = ex.StackTrace.ToString();

                                            Trace.WriteLine("Exception Stacktrace (2054): " + stackTrace, "INFO");

                                            continue;
                                        }
                                    }
                                }
                                asyncResult.TickerDataFields[ticker] = fieldList;
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing Reference Data OnProcessReferenceDataResponse: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace OnProcessReferenceDataResponse: " + stackTrace, "INFO");
            }
            //asyncResult.Completed();
        }

        private static void OnProcessHistoricalDataResponse(Message msg)
        {
            try
            {
                //Trace.WriteLine("OnProcessHistoricalDataResponse is called.", "TRACE");

                var asyncResult = (MarketDataResult)msg.CorrelationID.Object;

                if (msg.HasElement(BBCommMessageType.RESPONSEERROR))
                {
                    Trace.WriteLine(string.Format("Failed to process historical data to Bloomberg Data API Services: {0}", msg.GetElement(BBCommMessageType.RESPONSEERROR)), "ERROR");
                    return;
                }

                if (!msg.HasElement(Names.SECURITYDATA))
                {
                    Trace.WriteLine("OnProcessHistoricalDataResponse, message doesn't include securityData: " + msg.ToString() + "\n", "INFO");
                    return;
                }

                var securities = msg.GetElement(Names.SECURITYDATA);
                var ticker = securities.GetElementAsString(Names.SECURITY);
                var fieldDataArray = securities.GetElement(Names.FIELDDATA);

                for (int i = 0; i < fieldDataArray.NumValues; ++i)
                {
                    Element fieldData = fieldDataArray.GetValueAsElement(i);

                    if (fieldData != null)
                    {
                        List<IRateDataField> fieldList;

                        if (asyncResult.TickerRateDataFields.TryGetValue(ticker, out fieldList))
                        {
                            foreach (var field in fieldList)
                            {
                                var element = fieldData.GetElement(field.Name);

                                if (element != null)
                                {
                                    try
                                    {
                                        if (element.NumValues > 0)
                                            field.Value = element.GetValue();
                                        else
                                            field.Value = null;
                                    }
                                    catch (Exception ex)
                                    {
                                        Trace.WriteLine(string.Format("Error: (2055). Failed to retrieve historical data for Security {1} Field {2} Exception: {0}", ex.Message, ticker, field.Name), "ERROR");
                                        string stackTrace = "";
                                        if (ex.StackTrace != null)
                                            stackTrace = ex.StackTrace.ToString();

                                        Trace.WriteLine("Exception Stacktrace (2055): " + stackTrace, "INFO");
                                        continue;
                                    }
                                }

                                var dateElement = fieldData.GetElement(Names.DATE);
                                if (dateElement != null)
                                {
                                    try
                                    {
                                        if (dateElement.NumValues > 0)
                                            field.date = dateElement.GetValue();

                                    }
                                    catch (Exception ex)
                                    {
                                        Trace.WriteLine(string.Format("Error: (2056). Failed to retrieve historical data for Security {1} Field {2} Exception: {0}", ex.Message, ticker, Names.DATE.ToString()), "ERROR");
                                        string stackTrace = "";
                                        if (ex.StackTrace != null)
                                            stackTrace = ex.StackTrace.ToString();

                                        Trace.WriteLine("Exception Stacktrace (2056): " + stackTrace, "INFO");
                                        continue;
                                    }
                                }
                            }

                            asyncResult.TickerRateDataFields[ticker] = fieldList;
                        }
                    }
                }
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing Historical Data OnProcessHistoricalDataResponse: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace OnProcessHistoricalDataResponse: " + stackTrace, "INFO");
            }
            //asyncResult.Completed();
        }


        private static void OnProcessInstrumentListResponse(Message msg)
        {
            try
            {
                //Trace.WriteLine("OnProcessInstrumentListResponse is called.", "TRACE");

                var asyncResult = (MarketDataResult)msg.CorrelationID.Object;

                if (!msg.HasElement(Names.RESULTS))
                {
                    Trace.WriteLine("OnProcessInstrumentListResponse, message doesn't include results: " + msg.ToString() + "\n", "INFO");
                    return;
                }

                Element results = msg.GetElement(Names.RESULTS);

                if (results.NumValues > 0)
                {
                    List<SecurityDescription> SecurityDescList = new List<SecurityDescription>();
                    int numResults = results.NumValues;

                    for (int i = 0; i < numResults; ++i)
                    {
                        Element result = results.GetValueAsElement(i);

                        SecurityDescList.Add(new SecurityDescription(result.GetElementAsString(Names.SECURITY),
                                                                     result.GetElementAsString(Names.DESCRIPTION)));
                    }

                    if (asyncResult.DataFields[0].Name == Names.RESULTS.ToString() && SecurityDescList.Count > 0)
                        asyncResult.DataFields[0].Value = SecurityDescList;
                }
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing OnProcessInstrumentListResponse: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace OnProcessInstrumentListResponse: " + stackTrace, "INFO");
            }
            //asyncResult.Completed();
        }

        private static Element GetValues(Message msg)
        {
            try
            {
                if (msg.HasElement(BBCommMessageType.RESPONSEERROR))
                {
                    Trace.WriteLine(string.Format("Failed to get values (field data) from Bloomberg Data API Services: {0}", msg.GetElement(BBCommMessageType.RESPONSEERROR)), "ERROR");
                    return null;
                }

                var securities = msg.GetElement(Names.SECURITYDATA);

                var security = securities.GetValueAsElement();
                var ticker = security.GetElementAsString(Names.SECURITY);

                if (security.HasElement(Names.SECURITYERROR))
                {
                    Trace.WriteLine(string.Format("Failed to get values (field data) for Security {1} Exception: {0}", security.GetElement(Names.SECURITYERROR), ticker), "ERROR");
                    return null;
                }

                return security.GetElement(Names.FIELDDATA);
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing GetValues: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace GetValues: " + stackTrace, "INFO");
                return null;
            }
        }

        public IMarketDataResult GetValuesForFields(ITicker ticker, string field)
        {
            try
            {
                //Trace.WriteLine("GetValuesForFields is called for " + ticker + " - " + field, "TRACE");

                var refDataService = session.GetService("//blp/refdata");
                var request = refDataService.CreateRequest("ReferenceDataRequest");

                //creating asyncResult for correlation
                var asyncResult = new MarketDataResult(ticker, field);

                // Add ticker to request
                var securities = request.GetElement(Names.SECURITIES);
                securities.AppendValue(ticker.Ticker);

                // Add fields to request
                var reqFields = request.GetElement(Names.FIELDS);

                foreach (DataField fields in asyncResult.DataFields)
                    reqFields.AppendValue(fields.Name);

                session.SendRequest(request, new CorrelationID(asyncResult));

                ProcessSessionEvent(session);

                return asyncResult;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing GetValuesForFields: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace GetValuesForFields: " + stackTrace, "INFO");
                return null;
            }
        }

        public IMarketDataResult GetValuesForFields(IEnumerable<string> tickers, IEnumerable<string> fields)
        {
            try
            {
                //Trace.WriteLine("GetValuesForFields is called for ticker list - field list", "TRACE");

                if (tickers.Count() == 0 || fields.Count() == 0)
                    return null;

                if (tickers.Count() >= 10 && BLPSessionManager.getInstance().NumberOfConnection >= 2)
                {
                    Trace.WriteLine("GetValuesForFields is called for tickers & fields. Parallel sessions will work to retrieve data from Bloomberg. ()", "TRACE");

                    List<string> tickers1 = new List<string>();

                    int lastIndex = tickers.Count() / 3;

                    for (int i = 0; i < lastIndex; ++i)
                        tickers1.Add(tickers.ToList()[i]);

                    List<string> tickers2 = new List<string>();

                    for (int i = lastIndex; i < lastIndex * 2; ++i)
                        tickers2.Add(tickers.ToList()[i]);

                    List<string> tickers3 = new List<string>();

                    for (int i = lastIndex * 2; i < tickers.Count(); ++i)
                        tickers3.Add(tickers.ToList()[i]);

                    if (tickers1.Count == 0 || tickers2.Count == 0 || tickers3.Count == 0)
                        return null;

                    bool thread1Completed = true;
                    bool thread2Completed = true;

                    IMarketDataResult asyncResult1 = null;


                    Thread worker1 = new Thread(() =>
                    {
                        try
                        {
                            thread1Completed = false;
                            var session1 = BLPSessionManager.getInstance().getSession();

                            asyncResult1 = backworker(session1, tickers1, fields);
                            BLPSessionManager.getInstance().releaseSession(session1);
                            //Trace.WriteLine("thread1-session1 is completed.", "TRACE");
                        }
                        catch (Exception eWorker1)
                        {
                            string errMessage = "Bloomberg Api Market Data Service";

                            Trace.WriteLine("Error: (2058). Exception while retrieving data in parallel sessions (session 1): " + errMessage + "\n" + eWorker1.Message, "ERROR");

                            string stackTrace = "";
                            if (eWorker1.StackTrace != null)
                                stackTrace = eWorker1.StackTrace.ToString();

                            Trace.WriteLine("Exception Stacktrace (2058): " + stackTrace, "INFO");
                        }
                        finally
                        {
                            thread1Completed = true;
                        }
                    });
                    worker1.Start();


                    IMarketDataResult asyncResult2 = null;
                    Thread worker2 = new Thread(() =>
                    {
                        try
                        {
                            thread2Completed = false;
                            var session2 = BLPSessionManager.getInstance().getSession();

                            asyncResult2 = backworker(session2, tickers2, fields);
                            BLPSessionManager.getInstance().releaseSession(session2);
                        }
                        catch (Exception eWorker2)
                        {
                            string errMessage = "Bloomberg Api Market Data Service";

                            Trace.WriteLine("Error: (2058). Exception while retrieving data in parallel sessions (session 2): " + errMessage + "\n" + eWorker2.Message, "ERROR");

                            string stackTrace = "";
                            if (eWorker2.StackTrace != null)
                                stackTrace = eWorker2.StackTrace.ToString();

                            Trace.WriteLine("Exception Stacktrace (2058): " + stackTrace, "INFO");
                        }
                        finally
                        {
                            thread2Completed = true;
                        }
                        //Trace.WriteLine("thread2-session2 is completed.", "TRACE");
                    });
                    worker2.Start();

                    var asyncResult = backworker(session, tickers3, fields);

                    while (!thread1Completed || !thread2Completed)
                        System.Threading.Thread.Sleep(100);

                    if (asyncResult != null)
                    {
                        if (asyncResult1 != null)
                        {
                            foreach (var asyncRes in asyncResult1.TickerDataFields)
                                if (!asyncResult.TickerDataFields.ContainsKey(asyncRes.Key))
                                    asyncResult.TickerDataFields.Add(asyncRes.Key, asyncRes.Value);

                            asyncResult1.Dispose();
                        }

                        if (asyncResult2 != null)
                        {
                            foreach (var asyncRes in asyncResult2.TickerDataFields)
                                if (!asyncResult.TickerDataFields.ContainsKey(asyncRes.Key))
                                    asyncResult.TickerDataFields.Add(asyncRes.Key, asyncRes.Value);

                            asyncResult2.Dispose();
                        }
                    }

                    return asyncResult;
                }
                else
                    return backworker(session, tickers, fields);
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing GetValuesForFields: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace GetValuesForFields: " + stackTrace, "INFO");
                return null;
            }
        }

        private IMarketDataResult backworker(Session _session, IEnumerable<string> tickers, IEnumerable<string> fields)
        {
            try
            {
                var refDataService = _session.GetService("//blp/refdata");
                var request = refDataService.CreateRequest("ReferenceDataRequest");

                //creating asyncResult for correlation
                var asyncResult = new MarketDataResult(tickers, fields);

                // Add ticker to request
                var securities = request.GetElement(Names.SECURITIES);

                foreach (string ticker in asyncResult.TickerDataFields.Keys)
                    securities.AppendValue(ticker);

                // Add fields to request
                var reqFields = request.GetElement(Names.FIELDS);

                foreach (DataField field in asyncResult.TickerDataFields.First().Value)
                    reqFields.AppendValue(field.Name);

                _session.SendRequest(request, new CorrelationID(asyncResult));

                ProcessSessionEvent(_session);

                return asyncResult;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing backworker: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace backworker: " + stackTrace, "INFO");
                return null;
            }
        }

        public IMarketDataResult GetHistoricalValuesForFields(IEnumerable<string> tickers, IEnumerable<string> fields, string startdate, string enddate)
        {
            try
            {
                //Trace.WriteLine("GetHistoricalValuesForFields is called for date " + startdate, "TRACE");

                var refDataService = session.GetService("//blp/refdata");
                var request = refDataService.CreateRequest("HistoricalDataRequest");

                //creating asyncResult for correlation
                var asyncResult = new MarketDataResult(tickers, fields, startdate, enddate);

                // Add ticker to request
                var securities = request.GetElement(Names.SECURITIES);

                foreach (string ticker in asyncResult.TickerRateDataFields.Keys)
                    securities.AppendValue(ticker);

                // Add fields to request
                var reqFields = request.GetElement(Names.FIELDS);

                foreach (RateDataField field in asyncResult.TickerRateDataFields.First().Value)
                    reqFields.AppendValue(field.Name);

                //request.Set("periodicityAdjustment", "ACTUAL");
                request.Set("periodicitySelection", "DAILY");
                request.Set("startDate", startdate);
                request.Set("endDate", enddate);

                session.SendRequest(request, new CorrelationID(asyncResult));

                ProcessSessionEvent(session);

                return asyncResult;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing GetHistoricalValuesForFields: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace GetHistoricalValuesForFields: " + stackTrace, "INFO");
                return null;
            }
        }

        public IMarketDataResult GetQueryResults(ITicker ticker, string field)
        {
            try
            {
                //Trace.WriteLine("GetQueryResults is called for query " + ticker.Ticker, "TRACE");

                var instrumentService = session.GetService("//blp/instruments");
                var request = instrumentService.CreateRequest("instrumentListRequest");

                //creating asyncResult for correlation, ticker: queryText, fields: results
                var asyncResult = new MarketDataResult(ticker, field);

                request.Set(Names.QUERY, ticker.Ticker + " <equity>");

                request.Set(Names.MAXRESULTS, 10);

                session.SendRequest(request, new CorrelationID(asyncResult));

                ProcessSessionEvent(session);

                return asyncResult;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while processing GetQueryResults: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace GetQueryResults: " + stackTrace, "INFO");
                return null;
            }
        }

        public IMarketDataSubscription Subscribe(ITicker ticker, IEnumerable<string> fields)
        {
            try
            {
                var apiSubscription = new MarketDataSubscription(ticker, fields);
                sessionAsync.Subscribe(new List<Subscription> { apiSubscription.Subscription });

                return apiSubscription;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while Subscribe: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace Subscribe: " + stackTrace, "INFO");
                return null;
            }
        }

        public void Unsubscribe(IList<IMarketDataSubscription> marketDataSubscriptions)
        {
            try
            {
                if (marketDataSubscriptions == null)
                {
                    //throw new BlpApiDataException(mainThreadDispatcher, ("While UnSubscription, NULL argument - marketDataSubscriptions"));
                    Trace.WriteLine("While UnSubscription, NULL argument - marketDataSubscriptions", "WARNING");
                    return;
                }

                var subscriptions = new List<Subscription>();

                foreach (MarketDataSubscription marketSubscription in marketDataSubscriptions)
                    subscriptions.Add(((MarketDataSubscription)marketSubscription).Subscription);

                sessionAsync.Unsubscribe(subscriptions);
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while Unsubscribe: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace Unsubscribe: " + stackTrace, "INFO");
            }
        }

        public void Unsubscribe()
        {
            try
            {
                sessionAsync.Unsubscribe(new List<Subscription>(sessionAsync.GetSubscriptions()));
            }
            catch (Exception e)
            {
                Trace.WriteLine("Exception while Unsubscribe: " + e.Message, "ERROR");
                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace Unsubscribe: " + stackTrace, "INFO");
            }
        }

        #region IDisposable Members

        public void Dispose()
        {
            Trace.WriteLine("MarketDataProvider Dispose is called.", "TRACE");

            sessionAsync.Unsubscribe(new List<Subscription>(sessionAsync.GetSubscriptions()));
            sessionAsync.Stop();
            session.Stop();
            BLPSessionManager.getInstance().Dispose();
            ((IDisposable)sessionAsync).Dispose();
            ((IDisposable)session).Dispose();
            Logging.DeregisterCallback();

            Trace.WriteLine("MarketDataProvider Dispose is completed.", "TRACE");
        }

        #endregion
    }



}
