﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CDSHeatMap.Interface;
using CDSHeatMap.Model;
using CDSHeatMap.BloombergAPIManager;
using System.Collections;
using CDSHeatMap.Util;

using CDSHeatMap.ViewModel;
using System.Threading;
using System.Diagnostics;
using System.Windows.Threading;

namespace CDSHeatMap.BloombergDataManager
{
    public class DataManager : IDataManager
    {
        private readonly IMarketDataProvider _APIDataProvider;
        private readonly List<IMarketDataSubscription> _subscriptions = new List<IMarketDataSubscription>();

        public static event Action<int, string, string, double> SubscriptionResultUpdate;

        private static List<int> subscriptionResultList = new List<int>();

        public static string PricingSource { get; set; }

        public static int SubscribeCount = 0;

        public bool initialized = false;

        Dispatcher mainThreadDispatcher;

        public DataManager()
            : this(new MarketDataProvider())
        {        }

        public void Recover()
        { 
        
        }

        public DataManager(IMarketDataProvider apiDataProvider)
        {
            Trace.WriteLine("DataManager is created.", "TRACE");

            mainThreadDispatcher = Dispatcher.CurrentDispatcher;

            PricingSource = "Default";

            _APIDataProvider = apiDataProvider;

            _APIDataProvider.SubsriptionDataUpdate += UpdateCallBack;

            if (apiDataProvider.isInitialized()) initialized = true;

            Trace.WriteLine("DataManager creation is completed.", "TRACE");
        }

        public CheckPoint checkPointAll = new CheckPoint();
        public CheckPoint checkPointPart1 = new CheckPoint();
        public CheckPoint checkPointPart2 = new CheckPoint();

        public Dictionary<string, int> RetrieveCDSIndexSeries(List<string> securities)
        {
            Trace.WriteLine("RetrieveCDSIndexSeries is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            try
            {
                List<string> fields_CDSIndexSeries = new List<string> { DataFieldNames.ROLLING_SERIES };

                Dictionary<string, int> series = new Dictionary<string, int>();
                var asyncResult = _APIDataProvider.GetValuesForFields(securities, fields_CDSIndexSeries);

                foreach (string cdsindex_gen in securities)
                {
                    for (int i = 0; i < fields_CDSIndexSeries.Count; ++i)
                    {
                        if (asyncResult.TickerDataFields[cdsindex_gen][i].Name == DataFieldNames.ROLLING_SERIES)
                        {
                            if (asyncResult.TickerDataFields[cdsindex_gen][i].Value != null)
                            {
                                int serie;
                                if (int.TryParse((string)asyncResult.TickerDataFields[cdsindex_gen][i].Value, out serie))
                                    series.Add(cdsindex_gen, serie);
                                else
                                    series.Add(cdsindex_gen, -1);
                            }
                            else
                            {
                                if (cdsindex_gen.Contains("CDX HY B"))
                                    series.Add(cdsindex_gen, series["CDX HY CDSI GEN 5Y Corp"]);
                                else
                                    series.Add(cdsindex_gen, -1);
                            }
                        }
                    }
                }

                asyncResult.Dispose();

                Trace.WriteLine("RetrieveCDSIndexSeries is completed.", "TRACE");

                return series;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3010). Exception during RetrieveCDSIndexSeries function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3010): " + stackTrace, "INFO");
            }

            return null;
        }

        public Dictionary<string, string> RetrieveCDSIndexNames(List<string> securities)
        {
            Trace.WriteLine("RetrieveCDSIndexNames is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            try
            {
                List<string> fields_CDSIndexNames = new List<string> { DataFieldNames.NAME };

                Dictionary<string, string> cdsnames = new Dictionary<string, string>();
                var asyncResult = _APIDataProvider.GetValuesForFields(securities, fields_CDSIndexNames);

                foreach (string cdsindex_gen in securities)
                {
                    for (int i = 0; i < fields_CDSIndexNames.Count; ++i)
                    {
                        if (asyncResult.TickerDataFields[cdsindex_gen][i].Name == DataFieldNames.NAME)
                        {
                            cdsnames.Add(cdsindex_gen, (string)(asyncResult.TickerDataFields[cdsindex_gen][i].Value ?? ""));
                        }
                    }
                }

                asyncResult.Dispose();
                Trace.WriteLine("RetrieveCDSIndexNames is completed.", "TRACE");

                return cdsnames;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3001). Exception during RetrieveCDSIndexNames function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3001): " + stackTrace, "INFO");
            }

            return null;
        }


        public List<SecurityDescription> RetrieveInsQuery(string results, string query)
        {
            Trace.WriteLine("RetrieveInsQuery is called for " + query, "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            try
            {
                List<SecurityDescription> returnList = new List<SecurityDescription>();
                InsQuery insQuery = new InsQuery(query);

                var asyncResult = _APIDataProvider.GetQueryResults(insQuery, results);

                //asyncResult.AsyncWaitHandle.WaitOne();

                if (asyncResult.DataFields[0].Name == results)
                {
                    var securityList = asyncResult.DataFields[0].Value;

                    if (securityList != null)
                        foreach (SecurityDescription sec in (IList)securityList)
                        {
                            returnList.Add(new SecurityDescription(sec.Security, sec.Description));
                        }
                }

                asyncResult.Dispose();
                Trace.WriteLine("RetrieveInsQuery is completed for " + query, "TRACE");

                return returnList;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3002). Exception during RetrieveInsQuery function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3002): " + stackTrace, "INFO");
            }

            return null;
        }

        #region Subscription
        public void SubscribeForUpdates(CDSIndexMember security, CDSIndexType type, IEnumerable<string> fields)
        {
            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            if (fields == null || fields.Count() == 0)
                return;

            if (security.getTicker(type) != null)
            {
                security.Ticker = security.getTicker(type);
                if (type == CDSIndexType.EQUITY)
                {
                    security.Ticker = security.getTickerEquity();
                    if (security.Ticker != "")
                        _subscriptions.Add(_APIDataProvider.Subscribe(security, fields));
                }
                else if (type == CDSIndexType.SPREAD)
                {
                    security.Ticker = security.getTickerSpread(4, PricingSource);
                    if (security.Ticker != "")
                        _subscriptions.Add(_APIDataProvider.Subscribe(security, fields));
                }
            }
        }
                
        public void SubscribeForUpdates(ITicker security, IEnumerable<string> fields)
        {
            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            if (security.Ticker == "" || fields == null || fields.Count() == 0)
                return;

            _subscriptions.Add(_APIDataProvider.Subscribe(security, fields));
        }

        public void SubscribeForUpdates(List<ITicker> securityList, IEnumerable<string> fields)
        {
            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            if (fields == null || fields.Count() == 0)
                return;

            foreach (ITicker member in securityList)
            {
                if (member.Ticker != "")
                    _subscriptions.Add(_APIDataProvider.Subscribe(member, fields));
            }
        }

        public void Unsubscribe()
        {
            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            _APIDataProvider.Unsubscribe();
        }
        #endregion

        #region REtrievingInitialField
        public void GetCDSIndexMembers(CDSIndex _cdsIndex)
        {
            Trace.WriteLine("GetCDSIndexMembers is called for " + _cdsIndex.CDSIndexName, "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            try
            {
                var asyncResult = _APIDataProvider.GetValuesForFields(_cdsIndex, DataFieldNames.INDX_MEMBERS);
                if (asyncResult.DataFields[0].Name == DataFieldNames.INDX_MEMBERS)
                {
                    var memberList = asyncResult.DataFields[0].Value;

                    foreach (IndexMember member in (IList)memberList)
                    {
                        if (_cdsIndex.IndexType == CDSIndexType.SPREAD)
                        {
                            _cdsIndex.addCDSIndexMember(member.MemberTickerandExchangeCode, member.Column4);    /*memberName, spreadTicker*/
                        }
                        else if (_cdsIndex.IndexType == CDSIndexType.EQUITY)
                        {
                            _cdsIndex.addCDSIndexMember(member.MemberTickerandExchangeCode);    /*equityTicker*/
                        }
                    }

                    if (_cdsIndex.IndexType == CDSIndexType.SPREAD)
                    {
                        List<string> fields = new List<string>();
                        fields.Add(DataFieldNames.PRIMARY_EQUITY_TICKER);

                        var asyncEqTickerResult = _APIDataProvider.GetValuesForFields(_cdsIndex.getAllSpreadTicker(), fields);

                        foreach (CDSIndexMember member in _cdsIndex.CDSIndexMemberList)
                        {
                            var ticker = member.getTicker(CDSIndexType.SPREAD);
                            if (ticker != "")
                            {
                                for (int i = 0; i < fields.Count; ++i)
                                {
                                    if (asyncEqTickerResult.TickerDataFields[ticker][i].Name == DataFieldNames.PRIMARY_EQUITY_TICKER)
                                    {
                                        member.TickerEquity = ((string)(asyncEqTickerResult.TickerDataFields[ticker][i].Value ?? ""));
                                    }
                                }
                            }
                        }
                        asyncEqTickerResult.Dispose();
                    }
                    else if (_cdsIndex.IndexType == CDSIndexType.EQUITY)
                    {
                        List<string> fields = new List<string>();
                        fields.Add(DataFieldNames.SECURITY_NAME);

                        var asyncEqTickerResult = _APIDataProvider.GetValuesForFields(_cdsIndex.getAllEquityTicker(), fields);

                        foreach (CDSIndexMember member in _cdsIndex.CDSIndexMemberList)
                        {
                            var ticker = member.getTicker(CDSIndexType.EQUITY);
                            if (ticker != "")
                            {
                                for (int i = 0; i < fields.Count; ++i)
                                {
                                    if (asyncEqTickerResult.TickerDataFields[ticker][i].Name == DataFieldNames.SECURITY_NAME)
                                    {
                                        member.CDSRefName = ((string)(asyncEqTickerResult.TickerDataFields[ticker][i].Value ?? ""));
                                    }
                                }
                            }
                        }
                        asyncEqTickerResult.Dispose();
                    }
                }

                asyncResult.Dispose();

                Trace.WriteLine("GetCDSIndexMembers is completed for " + _cdsIndex.CDSIndexName, "TRACE");
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3003). Exception during GetCDSIndexMembers function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3003): " + stackTrace, "INFO");
            }
        }

        public void GetInitCDSIndex(CDSIndex _cdsIndex, List<string> AdditionalFields = null)
        {
            Trace.WriteLine("GetInitCDSIndex is called for " + _cdsIndex.CDSIndexName, "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            try
            {
                checkPointAll.reset();
                checkPointPart1.reset();
                checkPointPart2.reset();

                checkPointAll.enter();
                checkPointPart1.enter();
                #region RetrieveCDS_MEMBERS

                Trace.WriteLine("we are here GetInitCDSIndex", "TRACE");

                var asyncResult = _APIDataProvider.GetValuesForFields(_cdsIndex, DataFieldNames.INDX_MEMBERS);
                if (asyncResult.DataFields[0].Name == DataFieldNames.INDX_MEMBERS)
                {
                    var memberList = asyncResult.DataFields[0].Value;

                    foreach (IndexMember member in (IList)memberList)
                    {
                        if (_cdsIndex.IndexType == CDSIndexType.SPREAD)
                        {
                            _cdsIndex.addCDSIndexMember(member.MemberTickerandExchangeCode, member.Column4);    /*memberName, spreadTicker*/
                        }
                        else if (_cdsIndex.IndexType == CDSIndexType.EQUITY)
                        {
                            _cdsIndex.addCDSIndexMember(member.MemberTickerandExchangeCode);    /*equityTicker*/
                        }
                    }
                }
                #endregion
                checkPointPart1.exit();

                checkPointPart2.enter();
                if (_cdsIndex.IndexType == CDSIndexType.SPREAD)
                {
                    List<string> fieldsInit_SPREADTICKER = new List<string> { DataFieldNames.PRIMARY_EQUITY_TICKER, DataFieldNames.PRICING_SOURCE, DataFieldNames.CURRENCY };
                    GetInitSpreadFields(_cdsIndex.CDSIndexMemberList, fieldsInit_SPREADTICKER);
                    GetInitEquityFields(_cdsIndex.CDSIndexMemberList, AdditionalFields);
                }
                else if (_cdsIndex.IndexType == CDSIndexType.EQUITY)
                {
                    if (AdditionalFields == null)
                    {
                        AdditionalFields = new List<string>();
                    }
                    AdditionalFields.Add(DataFieldNames.SECURITY_NAME);

                    GetInitEquityFields(_cdsIndex.CDSIndexMemberList, AdditionalFields);

                    List<string> fieldsInit_SPREADTICKER = new List<string> {DataFieldNames.PRICING_SOURCE, DataFieldNames.CURRENCY };
                    GetInitSpreadFields(_cdsIndex.CDSIndexMemberList, fieldsInit_SPREADTICKER);
                }

                asyncResult.Dispose();
                GetInitEquitySpreadTickerPrice(_cdsIndex.CDSIndexMemberList);

                checkPointPart2.exit();
                checkPointAll.exit();

                Trace.WriteLine("GetInitCDSIndex is completed for " + _cdsIndex.CDSIndexName, "TRACE");

            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3004). Exception during GetInitCDSIndex function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3004): " + stackTrace, "INFO");
            }
        }

        private void GetInitEquityFields(List<CDSIndexMember> _CDSIndexMemberList, List<string> AdditionalFields = null)
        {
            Trace.WriteLine("GetInitEquityFields is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            try
            {
                List<string> fieldsInit_EQUITYTICKER = new List<string> { DataFieldNames.CDS_SPREAD_TICKER_1Y, DataFieldNames.CDS_SPREAD_TICKER_2Y, DataFieldNames.CDS_SPREAD_TICKER_3Y, 
                                                                          DataFieldNames.CDS_SPREAD_TICKER_4Y, DataFieldNames.CDS_SPREAD_TICKER_5Y, DataFieldNames.CDS_SPREAD_TICKER_7Y, 
                                                                          DataFieldNames.CDS_SPREAD_TICKER_10Y, DataFieldNames.RTG_MDY_SEN_UNSECURED_DEBT, DataFieldNames.RTG_SP_LT_LC_ISSUER_CREDIT, 
                                                                          DataFieldNames.RTG_FITCH_SEN_UNSECURED, DataFieldNames.INDUSTRY_SECTOR, DataFieldNames.LAST_PRICE, DataFieldNames.CUR_MKT_CAP, 
                                                                          DataFieldNames.TOT_DEBT_TO_TOT_CAP, DataFieldNames.TOT_DEBT_TO_COM_EQY, DataFieldNames.INTEREST_COVERAGE_RATIO, 
                                                                          DataFieldNames.CHG_PCT_1D, DataFieldNames.CHG_PCT_YTD, DataFieldNames.EQY_DVD_YLD_IND, DataFieldNames.ENTERPRISE_VALUE, 
                                                                          DataFieldNames.DDIS_AMT_OUTSTANDING, DataFieldNames.PE_RATIO, DataFieldNames.EBITDA, DataFieldNames.HIST_CALL_IMPL_VOL, 
                                                                          DataFieldNames.VOL_MEAN, DataFieldNames.BS_TOT_ASSET, DataFieldNames.SHORT_AND_LONG_TERM_DEBT, DataFieldNames.CURRENCY
                                                                       };

                if (AdditionalFields != null && AdditionalFields.Count > 0)
                {
                    foreach (string field in AdditionalFields)
                    {
                        fieldsInit_EQUITYTICKER.Add(field);
                    }
                }

                var asyncResult = _APIDataProvider.GetValuesForFields(getTickerList(_CDSIndexMemberList, CDSIndexType.EQUITY), fieldsInit_EQUITYTICKER);

                foreach (CDSIndexMember member in _CDSIndexMemberList)
                {
                    for (int i = 0; i < fieldsInit_EQUITYTICKER.Count; ++i)
                    {
                        if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.SECURITY_NAME.ToString())
                        {
                            member.CDSRefName = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_1Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_2Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_3Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_4Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_5Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));

                            if (member.TickerSpread == null || member.TickerSpread == "")
                                member.TickerSpread = (string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? "");

                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_7Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CDS_SPREAD_TICKER_10Y.ToString())
                        {
                            member.createTickerSpread((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }

                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.RTG_MDY_SEN_UNSECURED_DEBT.ToString())
                        {
                            member.MoodysRating = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ConstantItemList.MOODYSNAV));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.RTG_SP_LT_LC_ISSUER_CREDIT.ToString())
                        {
                            member.SPRating = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.RTG_FITCH_SEN_UNSECURED.ToString())
                        {
                            member.FitchRating = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.INDUSTRY_SECTOR.ToString())
                        {
                            member.IndustryName = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CURRENCY.ToString())
                        {
                            member.Currency = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value ?? ""));
                        }
                        else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value != null)
                        {
                            if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.LAST_PRICE.ToString())
                            {
                                member.LastPrice = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CUR_MKT_CAP.ToString())
                            {
                                member.MarketCapital = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.TOT_DEBT_TO_TOT_CAP.ToString())
                            {
                                member.TotalDebtToCapital = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.TOT_DEBT_TO_COM_EQY.ToString())
                            {
                                member.DebtToCommonEquity = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.INTEREST_COVERAGE_RATIO.ToString())
                            {
                                member.InterestCoverageRatio = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CHG_PCT_1D.ToString())
                            {
                                member.ChgPCT1D = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CHG_PCT_YTD.ToString())
                            {
                                member.ChgPCTYTD = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CHG_NET_1D.ToString())
                            {
                                member.ChgNET1D = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.CHG_NET_YTD.ToString())
                            {
                                member.ChgNETYTD = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.EQY_DVD_YLD_IND.ToString())
                            {
                                member.EqDividendeYield = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.ENTERPRISE_VALUE.ToString())
                            {
                                member.EnterpriseValue = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.DDIS_AMT_OUTSTANDING.ToString())
                            {
                                member.DebtOS = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.PE_RATIO.ToString())
                            {
                                member.PeRatio = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.EBITDA.ToString())
                            {
                                member.EBITDA = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            //additional fields
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.EXPECTED_REPORT_DT.ToString())
                            {
                                member.ExpectedReportDT = ((Bloomberglp.Blpapi.Datetime)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value)).ToSystemDateTime();
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.SI_PERCENT_EQUITY_FLOAT.ToString())
                            {
                                member.ShortInterest = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.PX_TO_SALES_RATIO.ToString())
                            {
                                member.PriceToSalesRatio = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.EQY_BETA.ToString())
                            {
                                member.EQBeta = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.SHORT_AND_LONG_TERM_DEBT.ToString())
                            {
                                member.TotalDebt = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.BS_TOT_ASSET.ToString())
                            {
                                member.TotalAssets = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.VOLUME_PCT_5_20.ToString())
                            {
                                member.VolumeChg5d20d = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.SALES_GROWTH.ToString())
                            {
                                member.SalesGrowth = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.HIST_CALL_IMPL_VOL.ToString())
                            {
                                member.HistCallImplVol = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Name == DataFieldNames.VOL_MEAN.ToString())
                            {
                                member.VolMean = (double)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                            }
                        }
                    }
                }

                asyncResult.Dispose();

                Trace.WriteLine("GetInitEquityFields is completed.", "TRACE");
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3005). Exception during GetInitEquityFields function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3005): " + stackTrace, "INFO");
            }
        }

        private void GetInitSpreadFields(List<CDSIndexMember> _CDSIndexMemberList, List<string> fieldsInit_SPREADTICKER)
        {
            Trace.WriteLine("GetInitSpreadFields is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            try
            {
                var asyncResult = _APIDataProvider.GetValuesForFields(getTickerList(_CDSIndexMemberList, CDSIndexType.SPREAD), fieldsInit_SPREADTICKER);

                foreach (CDSIndexMember member in _CDSIndexMemberList)
                {
                    if (member.getTicker(CDSIndexType.SPREAD) != "")
                    {
                        for (int i = 0; i < fieldsInit_SPREADTICKER.Count; ++i)
                        {
                            if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.SPREAD)][i].Name == DataFieldNames.PRIMARY_EQUITY_TICKER.ToString())
                            {
                                member.TickerEquity = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.SPREAD)][i].Value ?? ""));
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.SPREAD)][i].Name == DataFieldNames.PRICING_SOURCE.ToString())
                            {
                                member.PricingSource = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.SPREAD)][i].Value ?? ""));
                            }
                            else if (asyncResult.TickerDataFields[member.getTicker(CDSIndexType.SPREAD)][i].Name == DataFieldNames.CURRENCY.ToString())
                            {
                                member.CurrencyCDS = ((string)(asyncResult.TickerDataFields[member.getTicker(CDSIndexType.SPREAD)][i].Value ?? ""));
                            }
                        }
                    }

                }

                asyncResult.Dispose();

                Trace.WriteLine("GetInitSpreadFields is completed.", "TRACE");
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3006). Exception during GetInitSpreadFields function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3006): " + stackTrace, "INFO");
            }
        }

        public void GetInitEquitySpreadTickerPrice(List<CDSIndexMember> _CDSIndexMemberList, bool includeEquity = false)
        {
            Trace.WriteLine("GetInitEquitySpreadTickerPrice is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            try
            {
                List<string> fieldsInit_SPREADTICKERMAT = new List<string> { DataFieldNames.LAST_PRICE, DataFieldNames.CHG_PCT_1D, DataFieldNames.CHG_PCT_YTD };

                var asyncResult = _APIDataProvider.GetValuesForFields(getAllTickerList(_CDSIndexMemberList, includeEquity), fieldsInit_SPREADTICKERMAT);

                foreach (CDSIndexMember member in _CDSIndexMemberList)
                {
                    if (includeEquity)
                    {
                        var equityTicker = member.getTickerEquity();

                        if (equityTicker != "")
                        {
                            for (int j = 0; j < fieldsInit_SPREADTICKERMAT.Count; ++j)
                            {
                                if (asyncResult.TickerDataFields[equityTicker][j].Name == DataFieldNames.LAST_PRICE)
                                {
                                    if (asyncResult.TickerDataFields[equityTicker][j].Value != null)
                                    {
                                        double? rate = (double)(asyncResult.TickerDataFields[equityTicker][j].Value);
                                        member.LastPrice = rate;
                                    }
                                    else
                                        member.LastPrice = null;
                                }
                                else if (asyncResult.TickerDataFields[equityTicker][j].Name == DataFieldNames.CHG_PCT_1D)
                                {
                                    if (asyncResult.TickerDataFields[equityTicker][j].Value != null)
                                    {
                                        double? rate = (double)(asyncResult.TickerDataFields[equityTicker][j].Value);
                                        member.ChgPCT1D = rate;
                                    }
                                    else
                                        member.ChgPCT1D = null;
                                }
                                else if (asyncResult.TickerDataFields[equityTicker][j].Name == DataFieldNames.CHG_PCT_YTD)
                                {
                                    if (asyncResult.TickerDataFields[equityTicker][j].Value != null)
                                    {
                                        double? rate = (double)(asyncResult.TickerDataFields[equityTicker][j].Value);
                                        member.ChgPCTYTD = rate;
                                    }
                                    else
                                        member.ChgPCTYTD = null;
                                }
                            }
                        }
                        else
                        {
                            member.LastPrice = null;
                            member.ChgPCT1D = null;
                            member.ChgPCTYTD = null;
                        }
                    }

                    member.resetSpreadRateLists();

                    for (int i = 0; i < member.TickerSpreadList.Count; ++i)
                    {
                        string source = getPricingSource(member, PricingSource, i);

                        if (member.getTickerSpread(i) != "")
                        {
                            string spreadTicker;
                            if (source == "")
                                spreadTicker = member.getTickerSpread(i);
                            else
                                spreadTicker = member.getTickerSpread(i, source);

                            for (int j = 0; j < fieldsInit_SPREADTICKERMAT.Count; ++j)
                            {
                                if (asyncResult.TickerDataFields[spreadTicker][j].Name == DataFieldNames.LAST_PRICE.ToString())
                                {
                                    if (asyncResult.TickerDataFields[spreadTicker][j].Value != null)
                                    {
                                        double? rate = (double)(asyncResult.TickerDataFields[spreadTicker][j].Value);
                                        member.TickerSpreadRate(rate);
                                    }
                                    else
                                        member.TickerSpreadRate(null);
                                }
                                else if (asyncResult.TickerDataFields[spreadTicker][j].Name == DataFieldNames.CHG_PCT_1D.ToString())
                                {
                                    if (asyncResult.TickerDataFields[spreadTicker][j].Value != null)
                                    {
                                        double? rate = (double)(asyncResult.TickerDataFields[spreadTicker][j].Value);
                                        member.addChgPCT1D(rate);
                                    }
                                    else
                                        member.addChgPCT1D(null);
                                }
                                else if (asyncResult.TickerDataFields[spreadTicker][j].Name == DataFieldNames.CHG_PCT_YTD.ToString())
                                {
                                    if (asyncResult.TickerDataFields[spreadTicker][j].Value != null)
                                    {
                                        double? rate = (double)(asyncResult.TickerDataFields[spreadTicker][j].Value);
                                        member.addChgPCTYTD(rate);
                                    }
                                    else
                                        member.addChgPCTYTD(null);
                                }
                            }
                        }
                        else
                        {
                            member.TickerSpreadRate(null);
                            member.addChgPCT1D(null);
                            member.addChgPCTYTD(null);
                        }
                    }
                }

                asyncResult.Dispose();

                fieldsInit_SPREADTICKERMAT.Clear();

                GetInitEquitySpreadTickerNetChg(_CDSIndexMemberList);

                Trace.WriteLine("GetInitEquitySpreadTickerPrice is completed.", "TRACE");
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3007). Exception during GetInitEquitySpreadTickerPrice function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3007): " + stackTrace, "INFO");
            }
        }

        public void GetInitEquitySpreadTickerNetChg(List<CDSIndexMember> _CDSIndexMemberList)
        {
            Trace.WriteLine("GetInitEquitySpreadTickerNetChg is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return;
            }

            try
            {
                List<string> tickerlist = getTickerList(_CDSIndexMemberList, CDSIndexType.EQUITY);

                foreach(var member in _CDSIndexMemberList)
                {
                    string source = getPricingSource(member, PricingSource, 4);

                    if (member.getTickerSpread(4) != "")
                    {
                        string spreadTicker;
                        if (source == "")
                            spreadTicker = member.getTickerSpread(4);
                        else
                            spreadTicker = member.getTickerSpread(4, source);

                        tickerlist.Add(spreadTicker);
                    }
                }

                List<string> fieldsInit_SPREADTICKERNETCHG = new List<string> { DataFieldNames.CHG_NET_1D, DataFieldNames.CHG_NET_YTD };

                var asyncResult = _APIDataProvider.GetValuesForFields(tickerlist, fieldsInit_SPREADTICKERNETCHG);

                foreach (CDSIndexMember member in _CDSIndexMemberList)
                {
                    var equityTicker = member.getTickerEquity();

                    if (equityTicker != "")
                    {
                        for (int j = 0; j < fieldsInit_SPREADTICKERNETCHG.Count; ++j)
                        {
                            if (asyncResult.TickerDataFields[equityTicker][j].Name == DataFieldNames.CHG_NET_1D)
                            {
                                if (asyncResult.TickerDataFields[equityTicker][j].Value != null)
                                {
                                    double? rate = (double)(asyncResult.TickerDataFields[equityTicker][j].Value);
                                    member.ChgNET1D = rate;
                                }
                                else
                                    member.ChgNET1D = null;
                            }
                            else if (asyncResult.TickerDataFields[equityTicker][j].Name == DataFieldNames.CHG_NET_YTD)
                            {
                                if (asyncResult.TickerDataFields[equityTicker][j].Value != null)
                                {
                                    double? rate = (double)(asyncResult.TickerDataFields[equityTicker][j].Value);
                                    member.ChgNETYTD = rate;
                                }
                                else
                                    member.ChgNETYTD = null;
                            }
                        }
                    }
                    else
                    {
                        member.ChgNET1D = null;
                        member.ChgNETYTD = null;
                    }
               
                    string source = getPricingSource(member, PricingSource, 4);

                    if (member.getTickerSpread(4) != "")
                    {
                        string spreadTicker;
                        if (source == "")
                            spreadTicker = member.getTickerSpread(4);
                        else
                            spreadTicker = member.getTickerSpread(4, source);

                        for (int j = 0; j < fieldsInit_SPREADTICKERNETCHG.Count; ++j)
                        {
                            if (asyncResult.TickerDataFields[spreadTicker][j].Name == DataFieldNames.CHG_NET_1D.ToString())
                            {
                                if (asyncResult.TickerDataFields[spreadTicker][j].Value != null)
                                {
                                    double? rate = (double)(asyncResult.TickerDataFields[spreadTicker][j].Value);
                                    member.CDS5YChgNET1D = rate;
                                }
                                else
                                    member.CDS5YChgNET1D = null;
                            }
                            else if (asyncResult.TickerDataFields[spreadTicker][j].Name == DataFieldNames.CHG_NET_YTD.ToString())
                            {
                                if (asyncResult.TickerDataFields[spreadTicker][j].Value != null)
                                {
                                    double? rate = (double)(asyncResult.TickerDataFields[spreadTicker][j].Value);
                                    member.CDS5YChgNETYTD = rate;
                                }
                                else
                                    member.CDS5YChgNETYTD = null;
                            }
                        }
                    }
                    else
                    {
                        member.CDS5YChgNET1D = null;
                        member.CDS5YChgNETYTD = null;
                    }
                }

                asyncResult.Dispose();

                Trace.WriteLine("GetInitEquitySpreadTickerNetChg is completed.", "TRACE");
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3008). Exception during GetInitEquitySpreadTickerNetChg function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3008): " + stackTrace, "INFO");
            }
        }

        #endregion

        public Dictionary<string, Rate> RetrieveDatePriceForAll(List<CDSIndexMember> _CDSIndexMemberList, DateTime date1)
        {
            Trace.WriteLine("RetrieveDatePriceForAll is called for CDSIndexMember list and date  " + date1.ToString(), "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            Dictionary<string, Rate> returnHistList = new Dictionary<string, Rate>();

            try
            {
                List<string> fieldsALL_HistoricalPrice = new List<string> { DataFieldNames.PX_LAST };

                var asyncResult = _APIDataProvider.GetHistoricalValuesForFields(getAllTickerList(_CDSIndexMemberList, true), fieldsALL_HistoricalPrice,
                                                                                                    date1.ToString("yyyyMMdd"), date1.ToString("yyyyMMdd"));

                double? rate;

                foreach (CDSIndexMember member in _CDSIndexMemberList)
                {
                    if (member.getTickerEquity() != "")
                    {
                        for (int i = 0; i < fieldsALL_HistoricalPrice.Count; ++i)
                        {
                            if (asyncResult.TickerRateDataFields[member.getTickerEquity()][i].Name == DataFieldNames.PX_LAST.ToString())
                            {
                                if (asyncResult.TickerRateDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value != null)
                                    rate = (double)(asyncResult.TickerRateDataFields[member.getTicker(CDSIndexType.EQUITY)][i].Value);
                                else
                                    rate = null;
                                if (!returnHistList.ContainsKey(member.getTickerEquity()))
                                    returnHistList.Add(member.getTickerEquity(), new Rate(date1, rate));
                            }
                        }
                    }

                    for (int j = 0; j < member.TickerSpreadList.Count; ++j)
                    {
                        string source = getPricingSource(member, PricingSource, j);

                        if (member.getTickerSpread(j) != "")
                        {
                            for (int i = 0; i < fieldsALL_HistoricalPrice.Count; ++i)
                            {
                                string spreadTicker;
                                if (source == "")
                                    spreadTicker = member.getTickerSpread(j);
                                else
                                    spreadTicker = member.getTickerSpread(j, source);

                                if (asyncResult.TickerRateDataFields[spreadTicker][i].Name == DataFieldNames.PX_LAST.ToString())
                                {
                                    if (asyncResult.TickerRateDataFields[spreadTicker][i].Value != null)
                                        rate = (double)(asyncResult.TickerRateDataFields[spreadTicker][i].Value);
                                    else
                                        rate = null;

                                    if (!returnHistList.ContainsKey(spreadTicker))
                                        returnHistList.Add(spreadTicker, new Rate(date1, rate));
                                }
                            }
                        }
                    }
                }
                asyncResult.Dispose();

                Trace.WriteLine("RetrieveDatePriceForAll is completed for CDSIndexMember list and date  " + date1.ToString(), "TRACE");

            }
            catch (Exception e)
            {                
                Trace.WriteLine("Error: (3009). Exception during RetrieveDatePriceForAll function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3009): " + stackTrace, "INFO");
            }

            return returnHistList;
        }

        public Dictionary<string, Rate> RetrieveDatePriceForAll(IEnumerable<string> tickerList, DateTime date1)
        {
            Trace.WriteLine("RetrieveDatePriceForAll is called for tickerList and date " + date1.ToString(), "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            Dictionary<string, Rate> returnHistList = new Dictionary<string, Rate>();

            try
            {
                List<string> fields = new List<string>();

                if (DateTime.Now.Date == date1.Date)
                    fields.Add(DataFieldNames.LAST_PRICE);
                else
                    fields.Add(DataFieldNames.PX_LAST);

                var asyncResult = _APIDataProvider.GetHistoricalValuesForFields(tickerList, fields, date1.ToString("yyyyMMdd"), date1.ToString("yyyyMMdd"));

                double? rate;

                foreach (string ticker in tickerList)
                {
                    if (ticker == "")
                        continue;

                    for (int i = 0; i < fields.Count; ++i)
                    {
                        if (asyncResult.TickerRateDataFields[ticker][i].Name == DataFieldNames.PX_LAST.ToString())
                        {
                            if (asyncResult.TickerRateDataFields[ticker][i].Value != null)
                                rate = (double)(asyncResult.TickerRateDataFields[ticker][i].Value);
                            else
                                rate = null;

                            if (!returnHistList.ContainsKey(ticker))
                                returnHistList.Add(ticker, new Rate(date1, rate));
                        }
                        else if (asyncResult.TickerRateDataFields[ticker][i].Name == DataFieldNames.LAST_PRICE.ToString())
                        {
                            if (asyncResult.TickerRateDataFields[ticker][i].Value != null)
                                rate = (double)(asyncResult.TickerRateDataFields[ticker][i].Value);
                            else
                                rate = null;

                            if (!returnHistList.ContainsKey(ticker))
                                returnHistList.Add(ticker, new Rate(date1, rate));
                        }
                    }
                }
                asyncResult.Dispose();

                Trace.WriteLine("RetrieveDatePriceForAll is completed for tickerList and date " + date1.ToString(), "TRACE");
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3013). Exception during RetrieveDatePriceForAll function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3013): " + stackTrace, "INFO");
            }

            return returnHistList;
        }

        public Dictionary<string, List<IDataField>> GetFieldValues(IEnumerable<string> tickers, IEnumerable<string> fields)
        {
            Trace.WriteLine("GetFieldValues is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            try
            {
                var asyncResult = _APIDataProvider.GetValuesForFields(tickers, fields);

                Trace.WriteLine("GetFieldValues is completed.", "TRACE");
                return asyncResult.TickerDataFields;
            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3012). Exception during GetFieldValues function: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3012): " + stackTrace, "INFO");
            }

            return new Dictionary<string, List<IDataField>>();
        }

        public List<string> getTickerList(List<CDSIndexMember> _CDSIndexMemberList, CDSIndexType type)
        {
            Trace.WriteLine("getTickerList is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            List<string> tickerList = new List<string>();

            foreach (CDSIndexMember member in _CDSIndexMemberList)
            {
                if (member.getTicker(type) != "")
                {
                    tickerList.Add(member.getTicker(type));
                }
            }

            Trace.WriteLine("getTickerList is completed.", "TRACE");
            return tickerList;
        }

        public List<string> getAllTickerList(List<CDSIndexMember> _CDSIndexMemberList, bool includeEquity)
        {
            Trace.WriteLine("getAllTickerList is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            List<string> tickerList = new List<string>();

            foreach (CDSIndexMember member in _CDSIndexMemberList)
            {
                if (includeEquity && member.getTickerEquity() != "")
                    tickerList.Add(member.getTickerEquity());

                for (int i = 0; i < member.TickerSpreadList.Count; ++i)
                {
                    if (member.getTickerSpread(i) != "")
                    {
                        string source = getPricingSource(member, PricingSource, i);
                        if (source == "")
                            tickerList.Add(member.getTickerSpread(i));
                        else
                            tickerList.Add(member.getTickerSpread(i, source));
                    }
                }
            }

            Trace.WriteLine("getAllTickerList is completed.", "TRACE");
            return tickerList;
        }

        public List<string> getAllTickerList(CDSIndexMember member, bool includeEquity)
        {
            Trace.WriteLine("getAllTickerList is called.", "TRACE");

            if (initialized == false)
            {
                Trace.WriteLine("WARNING !!! Bloomberg Session couldn't be created!!!", "ERROR");
                Recover();
                if (initialized == false)
                    return null;
            }

            List<string> tickerList = new List<string>();

            if (includeEquity && member.getTickerEquity() != "")
                tickerList.Add(member.getTickerEquity());

            for (int i = 0; i < member.TickerSpreadList.Count; ++i)
            {
                if (member.getTickerSpread(i) != "")
                {
                    string source = getPricingSource(member, PricingSource, i);
                    if (source == "")
                        tickerList.Add(member.getTickerSpread(i));
                    else
                        tickerList.Add(member.getTickerSpread(i, source));
                }
            }

            Trace.WriteLine("getAllTickerList is completed.", "TRACE");
            return tickerList;
        }

        private string getDefaultPricingSource(string ccy)
        {
            if (ccy == "USD")
                return "CBGN";
            else if (ccy == "EUR" || ccy == "GBP" || ccy == "GBp")
                return "CBGL";
            else if (ccy == "JPY" || ccy == "AUD")
                return "CBGT";
            else
                return "CBGN";
        }

        private string getPricingSource(CDSIndexMember member, string PricingSource, int i)
        {
            string defaultsource = getDefaultPricingSource(member.Currency);

            if (member.PricingSource == null || member.PricingSource == "")
            {
                if (PricingSource == "Default")
                {
                    return defaultsource;
                }
                else
                    return PricingSource;
            }
            else if (PricingSource == "Default")
                return "";
            else
                return PricingSource;
        }

        private static void UpdateCallBack(IMarketDataSubscription apiDataSubscription)
        {
            try
            {
                var ticker = (DataChild)apiDataSubscription.Ticker;

                foreach (DataField field in apiDataSubscription.DataFields)
                {
                    if (field != null && field.Value != null)
                    {
                        double fieldValue = 0.0;

                        if (field.Value.GetType() == typeof(double))
                        {
                            fieldValue = (double)(field.Value);
                        }
                        else if (field.Value.GetType() == typeof(Single))
                        {
                            var fieldf = (Single)(field.Value);
                            fieldValue = Convert.ToDouble(fieldf);
                        }
                        else
                            continue;

                        if (SubscriptionResultUpdate != null)
                            SubscriptionResultUpdate(ticker.ID, apiDataSubscription.SubscriptionTicker, field.Name, fieldValue);
                    }
                }

            }
            catch (Exception e)
            {
                Trace.WriteLine("Error: (3011). Exception while processing subscription value: " + e.Message, "ERROR");

                string stackTrace = "";
                if (e.StackTrace != null)
                    stackTrace = e.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace (3011): " + stackTrace, "INFO");
            }

        }

        #region IDisposable Members

        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                // dispose managed resources
                _APIDataProvider.Dispose();
            }
            // free native resources
        }

        public void Dispose()
        {
            Trace.WriteLine("DataManager Dispose is called.", "TRACE");

            Dispose(true);
            GC.SuppressFinalize(this);

            Trace.WriteLine("DataManager Dispose is completed.", "TRACE");
        }

        #endregion

    }
}
