﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Diagnostics;
using System.Windows.Input;
using System.Windows.Media;

namespace CDSHeatMap.Model
{
    public class DataGroup : INotifyPropertyChanged
    {
        private int _ID;
        private string _groupName;
        private double _totalRectangleSize;
        private double _totalRectangleColor;
        private double _totalChildCircleColor;

        public Dictionary<string, double?> SelDataDisplayMin { get; set; }
        public Dictionary<string, double?> SelDataDisplayAvg { get; set; }
        public Dictionary<string, double?> SelDataDisplayMax { get; set; }
        public Dictionary<string, double>  SelDataChildCount { get; set; }

        private BooleanNotify _firstByGuidanceEnabled;

        public BooleanNotify FirstByGuidanceEnabled
        {
            get { return _firstByGuidanceEnabled; }
            set
            {
                _firstByGuidanceEnabled = value;
                OnPropertyChanged("FirstByGuidanceEnabled");
            }
        }
        
        public ObservableCollection<DataChildGroup> DataChildGroupCollection { get; set; }

        private ICommand _leftClickRect;
        public ICommand LeftClickRect
        {
            get { return _leftClickRect; }
            set
            {
                _leftClickRect = value;
                OnPropertyChanged("LeftClickRect");
            }
        }


        public int ID
        {
            get { return _ID; }
            set
            {
                _ID = value;
                OnPropertyChanged("ID");
            }
        }

        public string GroupName
        {
            get { return _groupName; }
            set
            {
                _groupName = value;
                OnPropertyChanged("GroupName");
            }
        }

        public double TotalRectangleSize
        {
            get { return _totalRectangleSize; }
            set
            {
                _totalRectangleSize = value;
                OnPropertyChanged("TotalRectangleSize");
            }
        }

        public double TotalRectangleColor
        {
            get { return _totalRectangleColor; }
            set
            {
                _totalRectangleColor = value;
                OnPropertyChanged("TotalRectangleColor");
            }
        }

        public double TotalChildCircleColor
        {
            get { return _totalChildCircleColor; }
            set
            {
                _totalChildCircleColor = value;
            }
        }

        public SolidColorBrush _groupBorderBrush { get; set; }
        public SolidColorBrush GroupBorderBrush
        {
            get { return _groupBorderBrush; }
            set
            {
                _groupBorderBrush = value;
                OnPropertyChanged("GroupBorderBrush");

                /* 21.05.2015 - bug. crash when app not run in LP
                if (_groupBorderBrush.CanFreeze)
                    _groupBorderBrush.Freeze();
                 */
            }
        }

        public void Dispose()
        {
            SelDataDisplayMin.Clear();
            SelDataDisplayAvg.Clear();
            SelDataDisplayMax.Clear();
            SelDataChildCount.Clear();
            DataChildGroupCollection.Clear();

        }

        public DataGroup(int id, string name)
        {
            DataChildGroupCollection = new ObservableCollection<DataChildGroup>();
            FirstByGuidanceEnabled = new BooleanNotify(false);

            ID = id;
            GroupName = name;

            TotalRectangleSize = TotalRectangleColor = TotalChildCircleColor = 0;

            SelDataDisplayMin = new Dictionary<string, double?>();
            SelDataDisplayAvg = new Dictionary<string, double?>();
            SelDataDisplayMax = new Dictionary<string, double?>();
            SelDataChildCount = new Dictionary<string, double>();
        }

        public DataGroup(int id,string name, DataChildGroup childgroup)
        {
            DataChildGroupCollection = new ObservableCollection<DataChildGroup>();
            FirstByGuidanceEnabled = new BooleanNotify(false);

            ID = id;
            GroupName = name;
            
            TotalRectangleSize = TotalRectangleColor = TotalChildCircleColor = 0;

            SelDataDisplayMin = new Dictionary<string, double?>();
            SelDataDisplayAvg = new Dictionary<string, double?>();
            SelDataDisplayMax = new Dictionary<string, double?>();
            SelDataChildCount = new Dictionary<string, double>();

            AddChildGroup(childgroup);
        }

        public void AddChildGroup(DataChildGroup child)
        {
            try
            {
                TotalRectangleSize += child.TotalChildRectangleSize;
                TotalRectangleColor += child.TotalChildRectangleColor;
                TotalChildCircleColor += child.TotalChildCircleColor;

                foreach (KeyValuePair<string, double?> key in child.SelDataDisplayMin)
                {
                    if (SelDataDisplayMin.ContainsKey((key.Key)))
                    {
                        if (SelDataDisplayMin[key.Key].HasValue && key.Value.HasValue)
                        {
                            if (SelDataDisplayMin[key.Key] > key.Value)
                            {
                                SelDataDisplayMin[key.Key] = key.Value;
                            }
                        }
                        else if (key.Value.HasValue)
                        {
                            SelDataDisplayMin[key.Key] = key.Value;
                        }
                    }
                    else
                    {
                        SelDataDisplayMin.Add(key.Key, key.Value);
                    }
                }

                foreach (KeyValuePair<string, double?> key in child.SelDataDisplayMax)
                {
                    if (SelDataDisplayMax.ContainsKey((key.Key)))
                    {
                        if (SelDataDisplayMax[key.Key].HasValue && key.Value.HasValue)
                        {
                            if (SelDataDisplayMax[key.Key] < key.Value)
                            {
                                SelDataDisplayMax[key.Key] = key.Value;
                            }
                        }
                        else if (key.Value.HasValue)
                        {
                            SelDataDisplayMax[key.Key] = key.Value;
                        }
                    }
                    else
                    {
                        SelDataDisplayMax.Add(key.Key, key.Value);
                    }
                }

                foreach (KeyValuePair<string, double?> key in child.SelDataDisplayAvg)
                {
                    if (SelDataDisplayAvg.ContainsKey((key.Key)))
                    {
                        if (key.Value.HasValue)
                        {
                            SelDataChildCount[key.Key] = SelDataChildCount[key.Key] + 1;
                            SelDataDisplayAvg[key.Key] = ((SelDataDisplayAvg[key.Key] * (SelDataChildCount[key.Key] - 1)) + key.Value) / SelDataChildCount[key.Key];
                        }
                    }
                    else
                    {
                        if (key.Value.HasValue)
                        {
                            SelDataChildCount.Add(key.Key, 1);
                            SelDataDisplayAvg.Add(key.Key, key.Value);
                        }
                    }
                }

                DataChildGroupCollection.Add(child);                
            }
            catch (Exception e)
            {
                Trace.WriteLine("DataChildGroupCollection.Add failed ", "TRACE");

                if(e.StackTrace != null)
                    Trace.WriteLine("DataChildGroupCollection.Add failed " + e.Message + "\n" + e.StackTrace, "TRACE");
            }
          }

        #region INotifyPropertyChanged
        public event PropertyChangedEventHandler PropertyChanged;

        private void OnPropertyChanged(string property)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new PropertyChangedEventArgs(property));
            }
        }
        #endregion

    }
}
