﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Input;
using CDSHeatMap.BloombergAPIManager;
using CDSHeatMap.BloombergDataManager;
using CDSHeatMap.Model;
using CDSHeatMap.Util;
using CDSHeatMap.Ticker;
using CDSHeatMap.View;
using System.Windows.Controls;
using System.Diagnostics;
using System.Threading;
using System.Globalization;

namespace CDSHeatMap.ViewModel
{
    class TickerFilterViewModel : BaseWindowViewModel
    {
        public override string WindowName
        {
            get { return "Ticker Filter"; }
        }

        public volatile bool IsBusy;

        public static bool isInitializedVM = false;

        public string savedListValue { get; set; }
        public string cdsIndexValue { get; set; }
        
        public ObservableCollection<IndustriesListDataObject> industriesValues { get; set; }
        public ObservableCollection<RatingsListDataObject> ratingsValues { get; set; }
        public string cdsPriceValue { get; set; }
        public static string regionValue { get; set; }
        public static Dictionary<string, List<string>> newList = new Dictionary<string, List<string>>();
        public ObservableCollection<IndustriesListDataObject> industriesList { get; set; }
        public ObservableCollection<RatingsListDataObject> ratingsList { get; set; }
        public ObservableCollection<string> cdsPriceList { get; set; }
        public ObservableCollection<CurveListDataObject> curveList { get; set; }
        public ObservableCollection<CheckedListDataObject> checkedList { get; set; }
        public ObservableCollection<string> savedList { get; set; }
        public List<string> regionList { get; set; }
        
        public TickerFilterViewModel()
        {
            Trace.WriteLine("TickerFilterViewModel is created.", "TRACE");

            savedList = new ObservableCollection<string>();
            ratingsList = new ObservableCollection<RatingsListDataObject>();
            industriesList = new ObservableCollection<IndustriesListDataObject>();
            curveList = new ObservableCollection<CurveListDataObject>();
            regionList = new List<string>();
            cdsPriceList = new ObservableCollection<string>();
            checkedList = new ObservableCollection<CheckedListDataObject>();

            if (!isDataManagerInitialized())
                return;

            initializeViewModel();

            Trace.WriteLine("TickerFilterViewModel is completed.", "TRACE");
        }

        public void initializeViewModel()
        {
            try
            {
                if (!isInitializedVM)
                {
                    isInitializedVM = true;
                    IsBusy = false;
                }
                fillInitialVMConstantLists();
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel initializeViewModel: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel initializeViewModel: " + stackTrace, "INFO");
            }
        }

        public void setFilterItemValues()
        {
            try
            {
                //regionValue = "ALL";
                fillRegionList();
                cdsIndexValue = getCDSIndexLongName(tickerFilterItems.CDSIndexName);
                cdsPriceValue = tickerFilterItems.PriceSource;
                savedListValue = tickerFilterItems.TickerFilterName;
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel setFilterItemValues: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel setFilterItemValues: " + stackTrace, "INFO");
            }
        }

        public string getCDSIndexLongName(string cdsindexname)
        {
            var cdsIndexList = cacheManager.getCDSIndexList();

            if (cdsIndexList == null)
            {
                Trace.WriteLine("Failed to retrieve CDS Index List while retrieving details of CDS Index Name " + cdsindexname, "ERROR");
                return null;
            }

            foreach (var cdsindex in cdsIndexList)
            {
                if (cdsindex.CDSIndexName == cdsindexname)
                {
                    return cdsindex.CDSIndexLongName;
                }
            }

            return cdsindexname;
        }

        public string getCDSIndexShortName(string cdsindexlongname)
        {
            var cdsIndexList = cacheManager.getCDSIndexList();

            if (cdsIndexList == null)
            {
                Trace.WriteLine("Failed to retrieve CDS Index List while retrieving details of CDS Index Name " + cdsindexlongname, "ERROR");
                return null;
            }

            foreach (var cdsindex in cdsIndexList)
            {
                if (cdsindex.CDSIndexLongName == cdsindexlongname)
                {
                    return cdsindex.CDSIndexName;
                }
            }

            return cdsindexlongname;
        }

        public void fillInitialVMConstantLists()
        {
            try
            {
                fillCurveList();
                fillCheckedList();
                fillRegionList();
                fillCdsPriceList();
                fillSavedList();
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel fillInitialVMConstantLists: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel fillInitialVMConstantLists: " + stackTrace, "INFO");
            }
        }

        public void fillInitialDynamicLists()
        {
            try
            {
                fillRatingsList();
                fillIndustriesList();
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel fillInitialDynamicLists: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel fillInitialDynamicLists: " + stackTrace, "INFO");
            }
        }

        public void loadInitialDynamicLists(TickerFilterItems tf)
        {
            try
            {
                loadIndustriesList(tf.Industries, tf.IndustriesChecked);
                loadRatingsList(tf.Ratings, tf.RatingsChecked);
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel loadInitialDynamicLists: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel loadInitialDynamicLists: " + stackTrace, "INFO");
            }
        }

        public Dictionary<string, List<string>> createRegionDict()
        {
            Dictionary<string, List<string>> mainList = new Dictionary<string, List<string>>();

            var cdsIndexList = cacheManager.getCDSIndexList();

            if (cdsIndexList == null)
            {
                Trace.WriteLine("Failed to retrieve CDS Index List while creating regions!", "ERROR");
                return null;
            }

            foreach (var cdsindex in cdsIndexList)
            {
                if (mainList.ContainsKey(cdsindex.region))
                {
                    if (!mainList[cdsindex.region].Contains(cdsindex.CDSIndexLongName))
                        mainList[cdsindex.region].Add(cdsindex.CDSIndexLongName);
                }
                else
                {
                    mainList.Add(cdsindex.region, new List<string> { cdsindex.CDSIndexLongName });
                }
            }
            return mainList;
        }

        public void fillRegionList()
        {
            try
            {
                regionList.Clear();
                newList.Clear();

                Dictionary<string, List<string>> regionDict = createRegionDict();

                if (regionDict == null)
                {
                    Trace.WriteLine("Failed to retrieve CDS Index region list!", "ERROR");
                    return;
                }

                RegionItems items = new RegionItems(regionDict);

                newList = ListUtil.addFirstItem("ALL", items.mainList);

                foreach (var reg in ListUtil.getAllKey(newList))
                    regionList.Add(reg);

                regionValue = regionList.First();
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel fillRegionList: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel fillRegionList: " + stackTrace, "INFO");
            }
        }

        public void fillCdsPriceList()
        {
            cdsPriceList.Clear();

            foreach (var priceSource in retrievePricingSourceList())
                cdsPriceList.Add(priceSource);

            cdsPriceValue = cdsPriceList.First();
        }

        public void fillSavedList()
        {
            savedList.Clear();
            var list = getTickerFilterList();

            if (list == null)
                return;

            foreach (var tf in list)
                savedList.Add(tf);
        }


        public void fillCurveList()
        {
            curveList.Clear();
            CurveListDataObject oneYear = new CurveListDataObject();
            oneYear.Name = "1Y/5Y";
            oneYear.IsChecked = true;
            curveList.Add(oneYear);

            CurveListDataObject twoYear = new CurveListDataObject();
            twoYear.Name = "2Y/5Y";
            twoYear.IsChecked = false;
            curveList.Add(twoYear);

            CurveListDataObject threeYear = new CurveListDataObject();
            threeYear.Name = "3Y/5Y";
            threeYear.IsChecked = false;
            curveList.Add(threeYear);

            CurveListDataObject sevenYear = new CurveListDataObject();
            sevenYear.Name = "5Y/7Y";
            sevenYear.IsChecked = false;
            curveList.Add(sevenYear);

            CurveListDataObject tenYear = new CurveListDataObject();
            tenYear.Name = "5Y/10Y";
            tenYear.IsChecked = false;
            curveList.Add(tenYear);
        }

        internal void checkCurveList(string curve, bool isChecked)
        {
            try
            {
                curveList.Clear();

                CurveListDataObject oneYear = new CurveListDataObject();
                oneYear.Name = "1Y/5Y";

                if (curve.Equals("1Y/5Y"))
                    oneYear.IsChecked = true;
                else
                    oneYear.IsChecked = false;
                curveList.Add(oneYear);

                CurveListDataObject twoYear = new CurveListDataObject();
                twoYear.Name = "2Y/5Y";
                if (curve.Equals("2Y/5Y"))
                    twoYear.IsChecked = true;
                else
                    twoYear.IsChecked = false;
                curveList.Add(twoYear);

                CurveListDataObject threeYear = new CurveListDataObject();
                threeYear.Name = "3Y/5Y";
                if (curve.Equals("3Y/5Y"))
                    threeYear.IsChecked = true;
                else
                    threeYear.IsChecked = false;
                curveList.Add(threeYear);

                CurveListDataObject sevenYear = new CurveListDataObject();
                sevenYear.Name = "5Y/7Y";
                if (curve.Equals("5Y/7Y"))
                    sevenYear.IsChecked = true;
                else
                    sevenYear.IsChecked = false;
                curveList.Add(sevenYear);

                CurveListDataObject tenYear = new CurveListDataObject();
                tenYear.Name = "5Y/10Y";

                if (curve.Equals("5Y/10Y"))
                    tenYear.IsChecked = true;
                else
                    tenYear.IsChecked = false;
                curveList.Add(tenYear);
            }
            catch (Exception e1)
            {
                Trace.WriteLine("Exception while TickerFilterViewModel checkCurveList: " + e1.Message, "ERROR");
                string stackTrace = "";
                if (e1.StackTrace != null)
                    stackTrace = e1.StackTrace.ToString();

                Trace.WriteLine("Exception Stacktrace TickerFilterViewModel checkCurveList: " + stackTrace, "INFO");
            }
        }

        public void fillCheckedList()
        {
            checkedList.Clear();
            CheckedListDataObject check = new CheckedListDataObject();
            check.Name = "Check All";
            check.IsChecked = true;
            checkedList.Add(check);

            CheckedListDataObject unCheck = new CheckedListDataObject();
            unCheck.Name = "Uncheck All";
            unCheck.IsChecked = false;
            checkedList.Add(unCheck);

            CheckedListDataObject removeUnCheck = new CheckedListDataObject();
            removeUnCheck.Name = "Remove Uncheck";
            removeUnCheck.IsChecked = false;
            checkedList.Add(removeUnCheck);
        }


        public void fillRatingsList()
        {
            ratingsList.Clear();
            foreach (var rating in retrieveRatingsList())
            {
                RatingsListDataObject obj = new RatingsListDataObject();
                obj.Name = rating;
                obj.IsChecked = true;
                ratingsList.Add(obj);
            }
        }

        internal bool getRatingsChecked(string tickerName)
        {
            if (!CacheManager.MoodysRatingCache.ContainsKey(tickerName + " Equity"))
                return false;

            string ratingName = CacheManager.MoodysRatingCache[tickerName + " Equity"].getData();

            foreach (var rating in ratingsList)
            {
                if (rating.Name == ratingName)
                    return rating.IsChecked;
            }

            return false;
        }

        internal void loadRatingsList(List<string> ratingList, List<bool> ratingChecked)
        {
            ratingsList.Clear();
            for (int i = 0; i < ratingList.Count; ++i)
            {
                RatingsListDataObject obj = new RatingsListDataObject();
                obj.Name = ratingList[i];
                obj.IsChecked = ratingChecked[i];
                ratingsList.Add(obj);
            }
        }

        internal void upd_loadRatingsList(List<string> ratingList, List<bool> ratingChecked)
        {
            ratingsList.Clear();
            for (int i = 0; i < ratingList.Count; ++i)
            {
                RatingsListDataObject obj = new RatingsListDataObject();
                obj.Name = ratingList[i];
                obj.IsChecked = ratingChecked[i];
                ratingsList.Add(obj);
            }

            foreach (var rating in retrieveRatingsList())
            {
                if (!ratingList.Contains(rating))
                {
                    RatingsListDataObject obj = new RatingsListDataObject();
                    obj.Name = rating;
                    obj.IsChecked = false;
                    ratingsList.Add(obj);
                }
            }
        }

        public void fillIndustriesList()
        {
            industriesList.Clear();
            foreach (var industry in retrieveIndustryList())
            {
                IndustriesListDataObject obj = new IndustriesListDataObject();
                obj.Name = industry;
                obj.IsChecked = true;
                industriesList.Add(obj);
            }
        }

        internal void loadIndustriesList(List<string> industryList, List<bool> industryChecked)
        {
            industriesList.Clear();
            for (int i = 0; i < industryList.Count; ++i)
            {
                IndustriesListDataObject obj = new IndustriesListDataObject();
                obj.Name = industryList[i];
                obj.IsChecked = industryChecked[i];
                industriesList.Add(obj);
            }
        }

        internal void upd_loadIndustriesList(List<string> industryList, List<bool> industryChecked)
        {
            industriesList.Clear();
            for (int i = 0; i < industryList.Count; ++i)
            {
                IndustriesListDataObject obj = new IndustriesListDataObject();
                obj.Name = industryList[i];
                obj.IsChecked = industryChecked[i];
                industriesList.Add(obj);
            }

            foreach (var industry in retrieveIndustryList())
            {
                if (!industryList.Contains(industry))
                {
                    IndustriesListDataObject obj = new IndustriesListDataObject();
                    obj.Name = industry;
                    obj.IsChecked = false;
                    industriesList.Add(obj);
                }
            }
        }

        internal bool getIndustriesChecked(string industryName)
        {
            foreach (var industry in industriesList)
            {
                if (industry.Name == industryName)
                    return industry.IsChecked;
            }
            return false;
        }

        #region RetrieveTickerFilterItems
        //retrive CDS Pricing Source List
        public List<string> retrievePricingSourceList()
        {
            return ConstantItemList.CDSPricingSourceList;
        }

        //retrieve Industry List
        public List<string> retrieveIndustryList()
        {
            cacheManager.CDSIndexIndustryList.Sort();
            return cacheManager.CDSIndexIndustryList;
        }

        //retrieve Moodys Rating List
        public List<string> retrieveRatingsList()
        {
            cacheManager.CDSIndexRatingsList.Sort();
            return cacheManager.CDSIndexRatingsList;
        }

        public string getCurveFilter_name()
        {
            foreach (var curve in curveList)
            {
                if (curve.IsChecked)
                {
                    return curve.Name;
                }
            }
            return "";
        }
        public List<double?> getCDSCurveFilter_MinMax()
        {
            foreach (var curve in curveList)
            {
                if (curve.Name == "1Y/5Y" && curve.IsChecked)
                {
                    return getCDSCurveFilterItem_MinMax(0, 4);
                }
                else if (curve.Name == "2Y/5Y" && curve.IsChecked)
                {
                    return getCDSCurveFilterItem_MinMax(1, 4);
                }
                else if (curve.Name == "3Y/5Y" && curve.IsChecked)
                {
                    return getCDSCurveFilterItem_MinMax(2, 4);
                }
                else if (curve.Name == "5Y/7Y" && curve.IsChecked)
                {
                    return getCDSCurveFilterItem_MinMax(4, 5);
                }
                else if (curve.Name == "5Y/10Y" && curve.IsChecked)
                {
                    return getCDSCurveFilterItem_MinMax(5, 6);
                }
            }

            return new List<double?>() { 0.0, 0.0 };
        }
        private List<double?> getCDSCurveFilterItem_MinMax(int curve1, int curve2)
        {
            List<double?> minmaxValue = new List<double?>() { null, null };

            foreach (CDSIndexMember member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
            {
                if (member.MidSpread == null)
                    continue;

                if (member.MidSpread.Count != 7)
                    continue;

                double? rate1 = member.getMidSpread(curve1);
                double? rate2 = member.getMidSpread(curve2);
                double rate;

                if (rate1.HasValue && rate2.HasValue && rate2 != 0.0)
                {
                    rate = (rate1 / rate2 ?? 0.0);
                }
                else
                    continue;


                if (minmaxValue[0].HasValue)
                {
                    if (rate < minmaxValue[0])
                        minmaxValue[0] = rate;
                }
                else
                    minmaxValue[0] = rate;

                if (minmaxValue[1].HasValue)
                {
                    if (rate > minmaxValue[1])
                        minmaxValue[1] = rate;
                }
                else
                    minmaxValue[1] = rate;

            }

            return minmaxValue;
        }

        //retrieve industry-ticker list
        public Dictionary<string, List<ItemChecked>> getIndustryTickerList()
        {
            Dictionary<string, List<ItemChecked>> industryTickerList = new Dictionary<string, List<ItemChecked>>();

            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
            {
                if (industryTickerList.ContainsKey(member.IndustryName))
                {
                    industryTickerList[member.IndustryName].Add(new ItemChecked(member.TickerEquity, member.IsCheckedTicker));
                }
                else
                {
                    List<ItemChecked> list = new List<ItemChecked>();
                    list.Add(new ItemChecked(member.TickerEquity, member.IsCheckedTicker));
                    industryTickerList.Add(member.IndustryName, list);
                }
            }

            if (tickerFilter.AddedTickerList.Count > 0)
            {
                List<ItemChecked> addedlist = new List<ItemChecked>();

                foreach (var ticker in tickerFilter.AddedTickerList)
                {
                    addedlist.Add(new ItemChecked(ticker.TickerEquity, ticker.IsCheckedTicker));
                }

                industryTickerList.Add("CUSTOM TICKER", addedlist);
            }

            return industryTickerList;
        }
        #endregion

        #region InstrumentSearch
        public List<SecurityDescription> searchTicker_SecDesc(string query)
        {
            return cacheManager.dataManager.RetrieveInsQuery(Names.RESULTS.ToString(), query);
        }

        public List<string> searchTicker(string query)
        {
            List<string> returnList = new List<string>();
            var list = cacheManager.dataManager.RetrieveInsQuery(Names.RESULTS.ToString(), query);

            if (list == null)
            {
                Trace.WriteLine("Failed to retrieve instrument list for security " + query, "INFO");
                return returnList;
            }

            foreach (var sec in list)
            {
                if (sec.Security.Contains(" <equity>"))
                {
                    sec.Security = sec.Security.Replace(" <equity>", "");
                }
                else if (sec.Security.Contains("<equity>"))
                {
                    sec.Security = sec.Security.Replace("<equity>", "");
                }

                returnList.Add(sec.Security + ":" + sec.Description);
            }

            return returnList;
        }

        public bool addTicker(SecurityDescription security)
        {
            if (!tickerFilter.AddedTickerList.Any(member => member.TickerEquity == security.Security))
            {
                tickerFilter.AddedTickerList.Add(new CDSIndexMember(security));

                tickerFilterItems.addedTickerList.Add(security.Security);
                tickerFilterItems.addedTickerDescriptionList.Add(security.Description);
                tickerFilterItems.filteredinTickerList.Add(security.Security);

                return true;
            }
            return false;
        }

        public void removeTicker(string ticker)
        {
            int index = tickerFilter.AddedTickerList.FindIndex(member => member.TickerEquity == ticker);
            if (index >= 0)
            {
                tickerFilterItems.addedTickerList.Remove(tickerFilter.AddedTickerList[index].TickerEquity);
                tickerFilterItems.addedTickerList.Remove(tickerFilter.AddedTickerList[index].CDSRefName);

                if (tickerFilterItems.unselectedTickerList.Contains(tickerFilter.AddedTickerList[index].TickerEquity))
                    tickerFilterItems.unselectedTickerList.Remove(tickerFilter.AddedTickerList[index].TickerEquity);

                tickerFilter.AddedTickerList.RemoveAt(index);
            }


        }

        #endregion

        #region TickerFilterItems
        public List<string> getTickerFilterList()
        {
            var list = cacheManager.getTickerFilterList();

            return list;
        }

        public bool getTickerFilterItems(string tickerFilterName)
        {
            var tfItem = cacheManager.getTickerFilterItems(tickerFilterName);
            if (tfItem == null)
                return false;
            else
            {
                tickerFilterItems = tfItem;
                return true;
            }
        }

        public bool loadTickerFilterItemsAndData(string tickerFilterName)
        {
            bool returnValue = true;
            //string errorMessage = "";

            returnValue = getTickerFilterItems(tickerFilterName);

            if (returnValue)
            {
                if (tickerFilterItems.CDSIndexName == "")
                {
                    //IsBusy = false;
                    return false;
                }

                var tickerCdsIndexBase = cacheManager.getCDSIndexBase(tickerFilterItems.CDSIndexName);

                if (tickerCdsIndexBase == null)
                {
                    //IsBusy = false;
                    //TODO: cds index not found

                    return false;
                }

                if (tickerCdsIndexBase.IndexType == CDSIndexType.EQUITY || tickerCdsIndexBase.IndexType == CDSIndexType.SPREAD)
                {
                    tickerFilter.reset();
                    tickerFilter.TickerFilterName = tickerFilterItems.TickerFilterName;

                    /*
                    initializeTickerFilter(new CDSIndexBase(tickerFilterItems.CDSIndexName, type));
                    getBloombergInitialDataForTickerFilter();
                    tickerFilter.uncheckCDSIndexMembers(tickerFilterItems.unselectedTickerList);
                    tickerFilter.uncheckCDSIndexMembers(tickerFilterItems.filteredoutTickerList);
                    */

                    cacheManager.initTickerFilter(tickerFilter, tickerCdsIndexBase, tickerFilterItems);

                    if (tickerFilter.addcoherentCustomTickers(tickerFilterItems))
                    {
                        cacheManager.saveTickerFilterItems(tickerFilterItems);
                    }

                    cacheManager.getInitialTickerFilterData(tickerFilter);

                    tickerFilter.checkAllTickers(tickerFilterItems);

                    //IsBusy = false;
                    return true;
                }
                else
                    tickerFilter.TickerFilterName = tickerFilterItems.TickerFilterName;
            }

            //IsBusy = false;

            return false;
        }

        public void addTickerFilterItems(string name)
        {
            tickerFilterItems.TickerFilterName = name;
            cacheManager.saveTickerFilterItems(tickerFilterItems);

            fillSavedList();
        }

        public void removeTickerFilterItems(string name)
        {
            cacheManager.removeTickerFilterItems(name);
            tickerFilterItems.TickerFilterName = "";
            fillSavedList();
        }
        #endregion

        #region BloombergDataRetrieval
        public void getBloombergInitialData(CDSIndexBase cdsIndexBase)
        {
            if (cdsIndexBase.IndexType == CDSIndexType.EQUITY)
            {
                tickerFilter.TFCDSIndex.reset();
                tickerFilter.TFCDSIndex.setCDSIndexName(cdsIndexBase);
                cacheManager.getInitialCDSIndex(tickerFilter.TFCDSIndex);
            }
            else if (cdsIndexBase.IndexType == CDSIndexType.SPREAD)
            {
                tickerFilter.TFCDSIndex.reset();
                tickerFilter.TFCDSIndex.setCDSIndexName(cdsIndexBase);
                cacheManager.getInitialCDSIndex(tickerFilter.TFCDSIndex);
            }

            //tickerFilter.TickerFilterName = "Current";
        }

        public void getBloombergInitialData(string cdsIndexName)
        {
            var tickerCdsIndexBase = cacheManager.getCDSIndexBase(cdsIndexName);

            if (tickerCdsIndexBase == null)
            {
                //TODO: cds index not found
                //IsBusy = false;
                return;
            }

            if (tickerCdsIndexBase.IndexType == CDSIndexType.UNKNOWN)
            {
                //IsBusy = false;
                return;
            }

            if (tickerCdsIndexBase.IndexType == CDSIndexType.EQUITY)
            {
                tickerFilter.TFCDSIndex.reset();
                tickerFilter.TFCDSIndex.setCDSIndexName(cdsIndexName, tickerCdsIndexBase.IndexType);
                cacheManager.getInitialCDSIndex(tickerFilter.TFCDSIndex);
            }
            else if (tickerCdsIndexBase.IndexType == CDSIndexType.SPREAD)
            {
                tickerFilter.TFCDSIndex.reset();
                tickerFilter.TFCDSIndex.setCDSIndexName(cdsIndexName, tickerCdsIndexBase.IndexType);
                cacheManager.getInitialCDSIndex(tickerFilter.TFCDSIndex);
            }

            //IsBusy = false;
            //tickerFilter.TickerFilterName = "Current";
        }

        public void updateCDSPricePriceSource()
        {
            cacheManager.dataManager.GetInitEquitySpreadTickerPrice(tickerFilter.getAllCDSMembers(), false);
            //IsBusy = false;
        }

        #endregion

        #region QuickFilterTicker

        public List<string> quickFilterTickers(List<QuickFilterViewObject> quickfilterItems)
        {
            List<string> FilteredTickerFalse = new List<string>();

            List<string> fields = new List<string>();
            foreach (var filter in quickfilterItems)
            {
                if (filter.fieldName != null && filter.fieldName != "")
                {
                    if (!filter.fieldName.Contains("CALCULATED"))
                        fields.Add(filter.fieldName);
                    else
                    {
                        string[] split = filter.fieldName.Split(new Char[] { '=', '/' });

                        foreach (string s in split)
                        {

                            if (s.Trim() != "" && s != "CALCULATED")
                                fields.Add(s);
                        }
                    }
                }
            }

            if (quickfilterItems.Count > 1)
                quickfilterItems[0].andOrValue = quickfilterItems[1].andOrValue;
            else
                quickfilterItems[0].andOrValue = "AND";

            if (fields.Count < 1)
                return null;

            var resultSet = cacheManager.dataManager.GetFieldValues(tickerFilter.TFCDSIndex.getAllEquityTicker(), fields);

            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
            {
                string ticker = member.getTickerEquity();
                if (ticker == "")
                    continue;

                if (resultSet.ContainsKey(ticker))
                {
                    bool isFilteredTicker = true;

                    for (int i = 0; i < quickfilterItems.Count; ++i)
                    {
                        bool isFilteredItem = true;
                        if (quickfilterItems[i].fieldName == null || quickfilterItems[i].fieldName == "")
                        {
                            continue;
                        }

                        if (quickfilterItems[i].fieldName.Contains("CALCULATED"))
                        {
                            string[] split = quickfilterItems[i].fieldName.Split(new Char[] { '=', '/' });
                            List<string> calcFields = new List<string>();

                            foreach (string s in split)
                            {
                                if (s.Trim() != "" && s != "CALCULATED")
                                    calcFields.Add(s);
                            }

                            double? value1 = null;
                            double? value2 = null;

                            if (calcFields.Count != 2)
                                continue;

                            for (int j = 0; j < resultSet.Values.Count; ++j)
                            {
                                if (resultSet[ticker][j].Value != null)
                                {
                                    if (resultSet[ticker][j].Name == calcFields[0])
                                    {
                                        value1 = (double)resultSet[ticker][j].Value;
                                    }

                                    if (resultSet[ticker][j].Name == calcFields[1])
                                    {
                                        value2 = (double)resultSet[ticker][j].Value;
                                    }
                                }
                                else
                                    isFilteredItem = false;
                            }

                            if (value1 != null && value2 != null && value2 != 0.0)
                            {
                                var val = value1 / value2;

                                if (CompareFilteredItems(val, quickfilterItems[i].operatorValue, quickfilterItems[i].minTextValue, quickfilterItems[i].maxTextValue))
                                {
                                    isFilteredItem = true;
                                }
                                else
                                {
                                    isFilteredItem = false;
                                }
                            }
                            else
                                isFilteredItem = false;
                        }
                        else
                        {

                            for (int j = 0; j < fields.Count; ++j)
                            {
                                if (resultSet[ticker][j].Name == quickfilterItems[i].fieldName)
                                {
                                    if (resultSet[ticker][j].Value != null)
                                    {
                                        var value = resultSet[ticker][j].Value;

                                        if (CompareFilteredItems(value, quickfilterItems[i].operatorValue, quickfilterItems[i].minTextValue, quickfilterItems[i].maxTextValue))
                                        {
                                            isFilteredItem = true;
                                        }
                                        else
                                        {
                                            isFilteredItem = false;
                                        }
                                    }
                                    else
                                        isFilteredItem = false;
                                }
                            }
                        }

                        if (i == 0)
                        {
                            isFilteredTicker = isFilteredItem;
                        }
                        else
                        {
                            if (quickfilterItems[i].andOrValue == "OR")
                                isFilteredTicker |= isFilteredItem;
                            else if (quickfilterItems[i].andOrValue == "AND")
                                isFilteredTicker &= isFilteredItem;

                        }
                    }

                    if (!isFilteredTicker)
                    {
                        FilteredTickerFalse.Add(member.TickerEquity);
                    }

                }
            }

            return FilteredTickerFalse;
        }

        public bool CompareFilteredItems(object itemValue, string compare, string filterValue, string maxFilterValue)
        {
            if (string.IsNullOrEmpty(filterValue))
                return false;

            switch (compare)
            {
                case "Between":
                    if (string.IsNullOrEmpty(maxFilterValue))
                        return false;

                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;
                        double valDouble2 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if (double.TryParse(maxFilterValue, NumberStyles.AllowDecimalPoint  | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble2))
                            {
                                if (((double)itemValue >= valDouble1 && ((double)itemValue <= valDouble2)))
                                    return true;
                                else
                                    return false;
                            }
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if ((((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date >= DateTime.Parse(filterValue).Date)
                            && (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date <= DateTime.Parse(maxFilterValue).Date))
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;
                        int valueInt2 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if (int.TryParse(maxFilterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt2))
                            {
                                if (((int)itemValue >= valueInt1) && ((int)itemValue <= valueInt2))
                                    return true;
                                else
                                    return false;
                            }
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    break;

                case "Equals":
                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if ((double)itemValue == valDouble1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        if (itemValue.ToString() == filterValue)
                            return true;
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date == DateTime.Parse(filterValue).Date)
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if ((int)itemValue == valueInt1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    break;

                case "Does Not Equal":
                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if ((double)itemValue != valDouble1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        if (itemValue.ToString() != filterValue)
                            return true;
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date != DateTime.Parse(filterValue).Date)
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if ((int)itemValue != valueInt1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    break;

                case "Greather Than":
                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if ((double)itemValue > valDouble1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date > DateTime.Parse(filterValue).Date)
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if ((int)itemValue > valueInt1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    break;

                case "Greather Than Or Equal To":
                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if ((double)itemValue >= valDouble1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date >= DateTime.Parse(filterValue).Date)
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if ((int)itemValue >= valueInt1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    break;

                case "Less Than":
                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if ((double)itemValue < valDouble1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date < DateTime.Parse(filterValue).Date)
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if ((int)itemValue < valueInt1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;

                    }
                    break;

                case "Less Than Or Equal To":
                    if (itemValue.GetType() == typeof(double))
                    {
                        double valDouble1 = 0.0;

                        if (double.TryParse(filterValue, NumberStyles.AllowDecimalPoint | NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valDouble1))
                        {
                            if ((double)itemValue <= valDouble1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    else if (itemValue.GetType() == typeof(string))
                    {
                        return false;
                    }
                    else if (itemValue.GetType() == typeof(Bloomberglp.Blpapi.Datetime))
                    {
                        if (((Bloomberglp.Blpapi.Datetime)itemValue).ToSystemDateTime().Date <= DateTime.Parse(filterValue).Date)
                            return true;
                        else
                            return false;

                    }
                    else if (itemValue.GetType() == typeof(Int32) || itemValue.GetType() == typeof(int))
                    {
                        int valueInt1 = 0;

                        if (int.TryParse(filterValue, NumberStyles.AllowLeadingSign, CultureInfo.InvariantCulture, out valueInt1))
                        {
                            if ((int)itemValue <= valueInt1)
                                return true;
                            else
                                return false;
                        }
                        else
                            return false;
                    }
                    break;

                case "Begins With":
                    if (itemValue.GetType() == typeof(string))
                    {
                        return itemValue.ToString().StartsWith(filterValue, StringComparison.CurrentCultureIgnoreCase);
                    }
                    break;

                case "Ends With":
                    if (itemValue.GetType() == typeof(string))
                    {
                        return itemValue.ToString().EndsWith(filterValue, StringComparison.CurrentCultureIgnoreCase);
                    }
                    break;

                case "Contains":
                    if (itemValue.GetType() == typeof(string))
                    {
                        if (itemValue.ToString().IndexOf(filterValue, StringComparison.CurrentCultureIgnoreCase) >= 0)
                            return true;
                        else return false;
                    }
                    break;

                case "Does Not Contain":
                    if (itemValue.GetType() == typeof(string))
                    {
                        if (itemValue.ToString().IndexOf(filterValue, StringComparison.CurrentCultureIgnoreCase) < 0)
                            return true;
                        else return false;
                    }
                    break;

                default:
                    return false;
            }

            return false;
        }

        #endregion

        internal List<string> retrieveTickerListWRTrating(string rating)
        {
            List<string> tickerList = new List<string>();
            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
                if (member.MoodysRating == rating)
                    tickerList.Add(member.TickerEquity);

            return tickerList;
        }

        internal double? getCDSSpread5Y(string ticker)
        {
            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
                if (member.TickerEquity == ticker)
                {
                    if (member.CDSPrice != null)
                        return member.CDSPrice * member.FXRateCDS;
                    else
                        return null;
                }

            return null;
        }

        internal double? getCDSCurve(string ticker)
        {
            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
                if (member.TickerEquity == ticker)
                {
                    foreach (var curve in curveList)
                    {
                        if (curve.Name == "1Y/5Y" && curve.IsChecked)
                        {
                            return getCDSCurveRate(member, 0, 4);
                        }
                        else if (curve.Name == "2Y/5Y" && curve.IsChecked)
                        {
                            return getCDSCurveRate(member, 1, 4);
                        }
                        else if (curve.Name == "3Y/5Y" && curve.IsChecked)
                        {
                            return getCDSCurveRate(member, 2, 4);
                        }
                        else if (curve.Name == "5Y/7Y" && curve.IsChecked)
                        {
                            return getCDSCurveRate(member, 4, 5);
                        }
                        else if (curve.Name == "5Y/10Y" && curve.IsChecked)
                        {
                            return getCDSCurveRate(member, 5, 6);
                        }
                    }
                }

            return null;
        }

        internal double? getCDSCurveRate(CDSIndexMember member, int curve1, int curve2)
        {
            if (member.MidSpread == null)
                return null;

            if (member.MidSpread.Count != 7)
                return null;

            double? rate1 = member.getMidSpread(curve1);
            double? rate2 = member.getMidSpread(curve2);

            if (rate1.HasValue && rate2.HasValue && rate2 != 0.0)
            {
                return (rate1 / rate2);
            }
            else
                return null;
        }

        internal double? getMarketCapital(string ticker)
        {
            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
                if (member.TickerEquity == ticker)
                {
                    if (member.MarketCapital != null)
                        return (member.MarketCapital * member.FXRateEquity);
                    else
                        return null;
                }

            return null;
        }

        internal double? getTotalDebtToCapital(string ticker)
        {
            if (CacheManager.TotalDebtToCapCache.ContainsKey(ticker + " Equity"))
                return CacheManager.TotalDebtToCapCache[ticker + " Equity"].getData();
            else
                return null;
        }

        internal double? getDebtToComEquity(string ticker)
        {
            if (CacheManager.DebtToCommonEqCache.ContainsKey(ticker + " Equity"))
                return CacheManager.DebtToCommonEqCache[ticker + " Equity"].getData();
            else
                return null;
        }

        internal double? getInterestCovRatio(string ticker)
        {
            if (CacheManager.InterestCovRatioCache.ContainsKey(ticker + " Equity"))
                return CacheManager.InterestCovRatioCache[ticker + " Equity"].getData();
            else
                return null;
        }

        internal CDSIndexBase getSelectedGroupType()
        {
            return HeatMapSelectedGroup;
        }

        internal string getPriceSource()
        {
            return DataManager.PricingSource;
        }

        internal void setPriceSource(string val)
        {
            DataManager.PricingSource = val;
        }

        internal TickerFilterItems getTickerFilterItems()
        {
            return tickerFilterItems;
        }

        internal TickerFilter getTickerFilter()
        {
            return tickerFilter;
        }

        internal void updateTickerFilterAfterFlag(bool value)
        {
            foreach (var ticker in tickerFilter.TFCDSIndex.CDSIndexMemberList)
            {
                ticker.IsCheckedTicker = value;
            }

            foreach (var ticker in tickerFilter.AddedTickerList)
            {
                ticker.IsCheckedTicker = value;
            }
        }

        internal bool getisTickerFilterChanged()
        {
            return isTickerFilterChanged;
        }

        internal void setisTickerFilterChanged(bool val)
        {
            isTickerFilterChanged = val;
        }

        internal bool getisCDSDataRetrieved()
        {
            return isCDSDataRetrieved;
        }

        internal void setisCDSDataRetrieved(bool val)
        {
            isCDSDataRetrieved = val;
        }

        internal bool getisAddedTickerDataRetrieved()
        {
            return isAddedTickerDataRetrieved;
        }

        internal void setisAddedTickerDataRetrieved(bool val)
        {
            isAddedTickerDataRetrieved = val;
        }

        internal void resetTickerFilterWithInitialData()
        {
            tickerFilter.AddedTickerList.Clear();
            tickerFilterItems.addedTickerList.Clear();
            tickerFilterItems.addedTickerDescriptionList.Clear();
            tickerFilterItems.selectedTickerList.Clear();
            tickerFilterItems.unselectedTickerList.Clear();
            tickerFilterItems.filteredoutTickerList.Clear();

            foreach (var member in tickerFilter.TFCDSIndex.CDSIndexMemberList)
                member.IsCheckedTicker = true;
        }

        public bool isDataManagerInitialized()
        {
            return cacheManager.dataManager.initialized;
        }

    }
}
