require('dotenv').config();
const {
    default: makeWASocket,
    useMultiFileAuthState,
    DisconnectReason,
    fetchLatestBaileysVersion
} = require('@whiskeysockets/baileys');
const { Boom } = require('@hapi/boom');
const { startAcademicNotificationsCron } = require('./cron/academicNotifications');
const qrcode = require('qrcode');
const path = require('path');
const fs = require('fs');
const assistantService = require('./services/assistantService');
const prisma = require("./lib/prisma");

// Función para manejar la respuesta del asistente
async function handleAssistantResponse(sock, from, response) {
    try {
        if (typeof response === 'string') {
            if (response.startsWith('http')) {
                await sock.sendMessage(from, {
                    image: { url: response },
                    caption: "¡Aquí está tu imagen! 🎨"
                });
            } else {
                await sock.sendMessage(from, { text: response });
            }
        } else {
            console.error('Respuesta inesperada del asistente:', response);
            throw new Error('Formato de respuesta inválido');
        }
    } catch (error) {
        console.error('Error al enviar respuesta:', error);
        throw error;
    }
}

// Función para manejar mensajes entrantes
async function handleIncomingMessage(sock, message, from, isGroup) {
    if (!message) return;

    const messageType = Object.keys(message)[0];
    let messageText = '';

    if (messageType === 'conversation') {
        messageText = message.conversation;
    } else if (messageType === 'extendedTextMessage') {
        messageText = message.extendedTextMessage.text;
    }

    // Manejar comandos específicos
    if (messageText.startsWith('!')) {
        await handleCommand(sock, messageText.toLowerCase(), from, isGroup);
        return;
    }

    try {
        // Procesar mensaje con el asistente
        const phoneNumber = from.split('@')[0];
        console.log(`💬 Procesando mensaje de ${phoneNumber}: ${messageText}`);

        // Enviar indicador de escritura
        await sock.sendPresenceUpdate('composing', from);

        const response = await assistantService.processMessage(phoneNumber, messageText);
        await handleAssistantResponse(sock, from, response);

        // Detener indicador de escritura
        await sock.sendPresenceUpdate('paused', from);

    } catch (error) {
        console.error('Error al procesar mensaje:', error);
        await sock.sendMessage(from, {
            text: "Lo siento, ocurrió un error al procesar tu mensaje. Por favor, intenta de nuevo en unos momentos. 🙏"
        });
    }
}

// Función para manejar comandos
async function handleCommand(sock, command, from, isGroup) {
    switch(command) {
        case '!ping':
            await sock.sendMessage(from, { text: 'pong 🏓' });
            break;

        case '!menu':
        case '!help':
        case '!ayuda':
            await sock.sendMessage(from, {
                text: `🤖 *MENÚ DE COMANDOS*\n\n` +
                      `!ping - Prueba de conexión\n` +
                      `!menu - Ver este menú\n` +
                      `!info - Información del bot\n\n` +
                      `💡 Para hablar conmigo o pedirme que genere imágenes,\n` +
                      `simplemente escribe tu mensaje normalmente.\n\n` +
                      `Ejemplos:\n` +
                      `• "Cuéntame sobre la historia de México"\n` +
                      `• "Genera una imagen de un atardecer en la playa"\n` +
                      `• "¿Cuál es la capital de Francia?"`
            });
            break;

        case '!info':
            await sock.sendMessage(from, {
                text: `ℹ️ *INFORMACIÓN DEL BOT*\n\n` +
                      `• Estado: Activo\n` +
                      `• Tipo de chat: ${isGroup ? 'Grupo' : 'Privado'}\n` +
                      `• Chat ID: ${from}\n` +
                      `• Asistente: GPT-4 + DALL-E 3\n` +
                      `• Versión: 1.0.0`
            });
            break;

        default:
            await sock.sendMessage(from, {
                text: '❌ Comando no reconocido\nUsa !menu para ver los comandos disponibles'
            });
    }
}

async function startWhatsAppBot() {
    try {
        // Inicializar el asistente
        await assistantService.initialize();

        // Crear directorio para el QR si no existe
        const qrPath = path.join(__dirname, 'qr');
        if (!fs.existsSync(qrPath)) {
            fs.mkdirSync(qrPath);
        }

        const { version } = await fetchLatestBaileysVersion();
        console.log(`Usando WA v${version.join('.')}`);

        const { state, saveCreds } = await useMultiFileAuthState(
            path.resolve(__dirname, './auth_info_baileys')
        );

        const sock = makeWASocket({
            version,
            printQRInTerminal: true,
            auth: state,
            browser: ['WhatsAppBot', 'Desktop', '3.0']
        });

        // Manejar conexión
        sock.ev.on('connection.update', async (update) => {
            const { connection, lastDisconnect, qr } = update;

            if (qr) {
                console.log('\n⚡ Nuevo código QR recibido');
                console.log('📱 Generando archivo QR...');

                try {
                    await qrcode.toFile(
                        path.join(qrPath, 'qr-code.png'),
                        qr,
                        {
                            width: 600,
                            margin: 3,
                            color: {
                                dark: '#000000',
                                light: '#ffffff'
                            }
                        }
                    );
                    console.log('✅ Código QR guardado en:', path.join(qrPath, 'qr-code.png'));
                    console.log('📱 Por favor, escanea el código QR con WhatsApp\n');
                } catch (qrError) {
                    console.error('❌ Error al generar QR:', qrError);
                }
            }

            if (connection === 'close') {
                const error = lastDisconnect?.error;
                const shouldReconnect = error instanceof Boom &&
                    error?.output?.statusCode !== DisconnectReason.loggedOut;

                console.log('Conexión cerrada debido a:', error?.output?.payload?.message || error);

                if (error?.output?.statusCode === DisconnectReason.loggedOut) {
                    console.log('❌ WhatsApp desvinculado. Eliminando credenciales...');
                    const authFolder = path.resolve(__dirname, './auth_info_baileys');
                    if (fs.existsSync(authFolder)) {
                        fs.rmSync(authFolder, { recursive: true, force: true });
                    }
                }

                if (shouldReconnect) {
                    console.log('🔄 Reconectando...');
                    setTimeout(startWhatsAppBot, 5000);
                } else {
                    console.log('🔄 Reiniciando bot para nuevo escaneo...');
                    setTimeout(startWhatsAppBot, 5000);
                }
            } else if (connection === 'open') {
                console.log('\n╔═══════════════════════════');
                console.log('║ BOT CONECTADO EXITOSAMENTE ✅');
                console.log('╚═══════════════════════════\n');

                // Iniciar el cron de notificaciones académicas
                startAcademicNotificationsCron(sock);
            }
        });

        // Guardar credenciales
        sock.ev.on('creds.update', saveCreds);

        // Manejar mensajes
        sock.ev.on('messages.upsert', async ({ messages, type }) => {
            if (type !== 'notify') return;

            const m = messages[0];
            if (!m.message || m.key.fromMe) return;

            // Ignorar mensajes de estado
            if (m.key.remoteJid === 'status@broadcast') return;

            const from = m.key.remoteJid;
            const isGroup = from.endsWith('@g.us');

            console.log('📩 Mensaje recibido:', {
                from: from,
                type: Object.keys(m.message)[0],
                isGroup: isGroup
            });

            await handleIncomingMessage(sock, m.message, from, isGroup);
        });

    } catch (error) {
        console.error('❌ Error al iniciar el bot:', error);
        console.log('🔄 Reintentando en 5 segundos...');
        setTimeout(startWhatsAppBot, 5000);
    }
}

// Limpiar la consola e iniciar
console.clear();
console.log('🚀 Iniciando WhatsApp Bot...\n');
startWhatsAppBot();