// src/services/academicTutorService.js - Nuevo archivo

const { Pool } = require('pg');
const { client, ASSISTANT_INSTRUCTIONS } = require('../config/openai'); // Añadir ASSISTANT_INSTRUCTIONS aquí
const userService = require('./database/userService');
const conversationService = require('./database/conversationService');
const messageService = require('./database/messageService');
const metricService = require('./database/metricService');

class AcademicTutorService {
    constructor() {
        this.academicPool = new Pool({
            connectionString: process.env.ACADEMIC_DATABASE_URL
        });
    }

    // Método para obtener estudiantes en riesgo
    async getStudentsAtRisk() {
        const query = `
            SELECT 
                c.identification_number AS dni,
                c.phone,
                c.email,
                concat(c.last_name1, ' ', c.last_name2, ' ', c.first_name) AS student_name,
                b.id AS enrollment_id,
                d.id AS syllabus_id,
                f.name AS period_name,
                e.name AS subject_name,
                g.name AS semester_name,
                h.name AS parallel_name,
                a.status AS prediction_status,
                a.status_bot,
                a.prediction_date
            FROM stacking_prediction a
            INNER JOIN enrollment b ON b.id = a.enrollment_id 
                AND b.status = 'ACTIVE'
            INNER JOIN student c ON c.id = b.student_id
            INNER JOIN syllabus d ON d.id = b.syllabus_id
            INNER JOIN subject e ON e.id = d.subject_id
            INNER JOIN period f ON f.id = d.period_id 
                AND f.status = 'A'
            INNER JOIN semester g ON g.id = d.semester_id
            INNER JOIN parallel h ON h.id = d.parallel_id
            WHERE a.status = 'FAILED' 
                AND a.status_bot = 0
            ORDER BY c.id`;

        try {
            const result = await this.academicPool.query(query);
            return result.rows;
        } catch (error) {
            console.error('Error obteniendo estudiantes en riesgo:', error);
            throw error;
        }
    }

    // Metodo para iniciar interacción con estudiante
    async initiateTutorSession(sock, studentData) {
        const startTime = Date.now();
        try {
            const user = await userService.getOrCreateUser(studentData.phone);
            const conversation = await conversationService.getOrCreateConversation(
                studentData.phone
            );

            // Generar mensaje inicial con métricas
            const { message, usage } = await this.generateInitialMessage(studentData);
            const responseTime = Math.floor(Date.now() - startTime);

            // Guardar mensaje con métricas completas
            await messageService.saveMessage({
                conversationId: conversation.conversation_id,
                phoneNumber: studentData.phone,
                messageType: 'assistant',
                content: message,
                modelUsed: 'gpt-4o-mini',
                tokensUsed: usage.total_tokens,
                completionTokens: usage.completion_tokens,
                promptTokens: usage.prompt_tokens,
                totalTokens: usage.total_tokens,
                responseTime: responseTime
            });

            // Enviar mensaje
            await sock.sendMessage(`${studentData.phone}@s.whatsapp.net`, {
                text: message
            });

            // Actualizar estado del estudiante
            await this.updateStudentStatus(studentData.enrollment_id);

            return {
                message,
                usage,
                responseTime
            };
        } catch (error) {
            console.error('Error iniciando sesión de tutoría:', error);
            throw error;
        }
    }
    // Metodo para generar mensaje inicial
    async generateInitialMessage(studentData) {
        const prompt = `
        Como tutor académico, genera un mensaje inicial para:
        Estudiante: ${studentData.student_name}
        Asignatura: ${studentData.subject_name}
        
        El mensaje debe ser:
        1. Personal y empático
        2. Mencionar específicamente la asignatura
        3. Ofrecer ayuda concreta
        4. Invitar a una respuesta
        5. Motivar al estudiante de manera positiva
        6. Sugerir un siguiente paso específico
        
        Reglas:
        - No mencionar que es un mensaje automático
        - No revelar acceso a datos académicos
        - Mantener un tono amigable pero profesional
        - Enfocarse en el apoyo y la mejora
        - Hacer el mensaje personal y relevante
        `;

        const completion = await client.chat.completions.create({
            model: "gpt-4o-mini",
            messages: [
                { role: "system", content: ASSISTANT_INSTRUCTIONS },
                { role: "user", content: prompt }
            ]
        });

        return {
            message: completion.choices[0].message.content,
            usage: completion.usage
        };
    }

    async updateStudentStatus(enrollmentId) {
        try {
            await this.academicPool.query(
                `UPDATE stacking_prediction 
                 SET status_bot = 1, 
                     notification_date = CURRENT_TIMESTAMP 
                 WHERE enrollment_id = $1`,
                [enrollmentId]
            );
        } catch (error) {
            console.error('Error actualizando estado del estudiante:', error);
            throw error;
        }
    }
}

// Exportar una única instancia del servicio
module.exports = new AcademicTutorService();