const { client, ASSISTANT_INSTRUCTIONS, ASSISTANT_TOOLS } = require('../config/openai');
const userService = require('./database/userService');
const conversationService = require('./database/conversationService');
const messageService = require('./database/messageService');
const metricService = require('./database/metricService');
const imageService = require('./database/imageService');
const prisma = require('../lib/prisma');
const taskService = require('./database/taskService');

class AssistantService {
    constructor() {
        this.assistant = null;
        this.userThreads = new Map(); // Almacena threads por número de teléfono
        this.vectorStoreId = 'vs_IZoJigzVmXNlNc901PbNjYCE'; // Tu vector store ID
    }

    async initialize() {
        try {
            // Crear o recuperar el asistente
            const assistants = await client.beta.assistants.list();
            this.assistant = assistants.data.find(a => a.name === "WhatsAppBot");

            if (!this.assistant) {
                this.assistant = await client.beta.assistants.create({
                    name: "WhatsAppBot",
                    instructions: ASSISTANT_INSTRUCTIONS,
                    model: "gpt-4o-mini",
                    tools: ASSISTANT_TOOLS,
                    tool_resources: {
                        file_search: {
                            vector_store_ids: [this.vectorStoreId]
                        }
                    }
                });
            } else {
                // Actualizar el asistente existente con el vector store
                this.assistant = await client.beta.assistants.update(
                    this.assistant.id,
                    {
                        instructions: ASSISTANT_INSTRUCTIONS,
                        tools: ASSISTANT_TOOLS,
                        tool_resources: {
                            file_search: {
                                vector_store_ids: [this.vectorStoreId]
                            }
                        }
                    }
                );
            }

            console.log('✅ Asistente inicializado correctamente');
            console.log(`🤖 ID del Asistente: ${this.assistant.id}`);
        } catch (error) {
            console.error('❌ Error al inicializar asistente:', error);
            throw error;
        }
    }


    async getOrCreateThread(phoneNumber) {
        try {
            if (!this.userThreads.has(phoneNumber)) {
                const thread = await client.beta.threads.create();
                this.userThreads.set(phoneNumber, thread.id);
                console.log(`📱 Nuevo hilo creado para: ${phoneNumber}`);
            }
            return this.userThreads.get(phoneNumber);
        } catch (error) {
            console.error(`❌ Error con el hilo de ${phoneNumber}:`, error);
            throw error;
        }
    }

    // En el metodo processMessage del AssistantService
    async processMessage(phoneNumber, message) {
        const startTime = Date.now();
        let apiCalls = 0;
        let errorCount = 0;
        let rateLimitHits = 0;

        try {
            // Obtener o crear usuario
            const user = await userService.getOrCreateUser(phoneNumber);

            // Obtener o crear thread
            const threadId = await this.getOrCreateThread(phoneNumber);

            // Obtener o crear conversación
            const conversation = await conversationService.getOrCreateConversation(phoneNumber, threadId);

            console.log(`✅ Conversación obtenida: ${conversation.conversation_id}`);

            // Obtener contexto reciente
            const recentMessages = await messageService.getRecentMessages(conversation.conversation_id, 4);
            console.log(`📝 Mensajes recientes recuperados: ${recentMessages.length}`);

            // Guardar mensaje del usuario
            const userMessage = await messageService.saveMessage({
                conversationId: conversation.conversation_id,
                phoneNumber,
                messageType: 'user',
                content: message
            });

            apiCalls++;

            // Procesar con OpenAI
            const response = await this.processWithOpenAI(
                threadId,
                message,
                conversation.conversation_id,
                userMessage.message_id,
                recentMessages
            );

            return response.content;

        } catch (error) {
            errorCount++;
            console.error('Error procesando mensaje:', error);
            throw error;
        }
    }

    async processWithOpenAI(threadId, message, conversationId, messageId) {
        try {

            // Obtener la conversación para conseguir el phone_number
            const conversation = await prisma.conversation.findUnique({
                where: { conversation_id: conversationId }
            });

            if (!conversation) {
                throw new Error('Conversación no encontrada');
            }

            // Obtener información del estudiante
            const studentInfo = await taskService.getStudentInfo(conversation.phone_number);
            console.log('📚 Información del estudiante:', studentInfo);


            // Construir el mensaje con contexto
            let contextMessage = message;
            if (studentInfo.success) {
                contextMessage = `[INFORMACIÓN DEL ESTUDIANTE]
                Nombre: ${studentInfo.student.nombre}
                Semestre: ${studentInfo.student.semester}
                Paralelo: ${studentInfo.student.parallel}
                Carrera: ${studentInfo.student.carrera}
                [FIN INFORMACIÓN]                
                Mensaje del estudiante: ${message}`;
            }

            // Obtener los 4 mensajes más recientes
            const previousMessages = await messageService.getConversationContext(conversationId);

            // Invertir el orden para mantener la cronología (del más antiguo al más reciente)
            const orderedMessages = previousMessages.reverse();

            // Crear mensaje en el thread con contexto
            for (const prevMsg of orderedMessages) {
                await client.beta.threads.messages.create(
                    threadId,
                    {
                        role: prevMsg.message_type === 'user' ? "user" : "assistant",
                        content: prevMsg.content
                    }
                );
            }

            // Agregar el mensaje actual
            await client.beta.threads.messages.create(
                threadId,
                { role: "user", content: contextMessage  }
            );

            // Ejecutar el asistente
            const run = await client.beta.threads.runs.create(
                threadId,
                { assistant_id: this.assistant.id }
            );

            // Esperar y procesar respuesta
            return await this.handleRunResponse(threadId, run.id, conversationId, messageId);

        } catch (error) {
            console.error('Error en processWithOpenAI:', error);
            throw error;
        }
    }

    async handleRunResponse(threadId, runId, conversationId, messageId) {
        const startTime = Date.now();
        while (true) {

            const run = await client.beta.threads.runs.retrieve(threadId, runId);

            switch (run.status) {
                case 'completed':
                    const messages = await client.beta.threads.messages.list(threadId);
                    const lastMessage = messages.data[0];

                    // Obtener la conversación para conseguir el phone_number
                    const conversation = await prisma.conversation.findUnique({
                        where: {
                            conversation_id: conversationId
                        }
                    });


                    if (!conversation) {
                        throw new Error('Conversación no encontrada');
                    }

                    const responseTime = Math.floor(Date.now() - startTime);

                    // Guardar respuesta del asistente
                    const assistantMessage = await messageService.saveMessage({
                        conversationId: conversationId,
                        phoneNumber: conversation.phone_number, // Usar el phone_number de la conversación
                        messageType: 'assistant',
                        content: lastMessage.content[0].text.value,
                        tokensUsed: run.usage?.total_tokens,
                        modelUsed: this.assistant.model,
                        completionTokens: run.usage?.completion_tokens,
                        promptTokens: run.usage?.prompt_tokens,
                        totalTokens: run.usage?.total_tokens,
                        responseTime: responseTime
                    });

                    return {
                        content: lastMessage.content[0].text.value,
                        totalTokens: run.usage?.total_tokens || 0,
                        responseTime
                    };

                case 'requires_action':
                    await this.handleFunctionCalls(threadId, runId, run.required_action, conversationId, messageId);
                    break;

                case 'failed':
                    throw new Error('Run failed: ' + run.last_error?.message);

                case 'expired':
                    throw new Error('Run expired');

                default:
                    await new Promise(resolve => setTimeout(resolve, 1000));
            }
        }
    }

    async handleFunctionCalls(threadId, runId, requiredAction, conversationId, messageId) {
    try {
        const toolOutputs = [];

        for (const toolCall of requiredAction.submit_tool_outputs.tool_calls) {
            switch (toolCall.function.name) {
                case 'dalle_generate_image':
                    console.log('🎨 Procesando llamada a DALL-E');
                    const { prompt } = JSON.parse(toolCall.function.arguments);

                    // Generar imagen
                    const imageUrl = await this.generateImage(prompt);
                    console.log('🖼️ URL de imagen generada:', imageUrl);

                    // Guardar imagen en la base de datos
                    try {
                        await imageService.saveGeneratedImage({
                            messageId: messageId,
                            conversationId: conversationId,
                            prompt: prompt,
                            imageUrl: imageUrl,
                            model: 'dall-e-3',
                            size: '1024x1024',
                            quality: 'standard'
                        });
                    } catch (error) {
                        console.error('❌ Error al guardar imagen:', error);
                    }

                    toolOutputs.push({
                        tool_call_id: toolCall.id,
                        output: imageUrl
                    });
                    break;

                case 'get_chat_response':
                    console.log('💬 Procesando respuesta de chat');
                    const { user_input } = JSON.parse(toolCall.function.arguments);
                    toolOutputs.push({
                        tool_call_id: toolCall.id,
                        output: JSON.stringify({ response: user_input })
                    });
                    break;

                case 'get_student_tasks':
                    console.log('📚 Procesando solicitud de tareas');

                    // Obtener la conversación para conseguir el phone_number
                    const conversation = await prisma.conversation.findUnique({
                        where: {
                            conversation_id: conversationId
                        }
                    });

                    if (!conversation) {
                        console.error('❌ Conversación no encontrada');
                        toolOutputs.push({
                            tool_call_id: toolCall.id,
                            output: JSON.stringify({
                                error: true,
                                mensaje: "Error al obtener información del estudiante"
                            })
                        });
                        break;
                    }

                    console.log('📱 Número de teléfono:', conversation.phone_number);

                    const tasksResult = await taskService.getStudentTasks(conversation.phone_number);
                    console.log('📊 Resultado de tareas:', JSON.stringify(tasksResult, null, 2));

                    if (tasksResult.success) {
                        const formattedTasks = this.formatTasksResponse(tasksResult.tasks);
                        console.log('✅ Tareas formateadas:', JSON.stringify(formattedTasks, null, 2));
                        toolOutputs.push({
                            tool_call_id: toolCall.id,
                            output: JSON.stringify(formattedTasks)
                        });
                    }  else {
                        console.log('❌ No se encontraron tareas:', tasksResult.message);
                        toolOutputs.push({
                            tool_call_id: toolCall.id,
                            output: JSON.stringify({
                                success: false,
                                mensaje: tasksResult.message || "No se encontraron tareas para este estudiante"
                            })
                        });
                    }
                    break;

                default:
                    console.warn(`⚠️ Función no reconocida: ${toolCall.function.name}`);
                    toolOutputs.push({
                        tool_call_id: toolCall.id,
                        output: JSON.stringify({
                            error: true,
                            mensaje: "Función no soportada"
                        })
                    });
            }
        }

        // Enviar todas las respuestas de las herramientas
        await client.beta.threads.runs.submitToolOutputs(
            threadId,
            runId,
            { tool_outputs: toolOutputs }
        );
    } catch (error) {
        console.error('❌ Error en handleFunctionCalls:', error);
        throw error;
    }
}

    async generateImage(prompt) {
        try {
            console.log('🎨 Generando imagen para prompt:', prompt);

            const response = await client.images.generate({
                model: "dall-e-3",
                prompt: prompt,
                size: "1024x1024",
                quality: "standard",
                n: 1
            });

            console.log('✅ Imagen generada exitosamente');
            return response.data[0].url;
        } catch (error) {
            console.error('❌ Error generando imagen:', error);
            throw error;
        }
    }

    // Agregar metodo auxiliar para formatear la respuesta de las tareas
    formatTasksResponse(tasks) {
        if (!tasks || tasks.length === 0) {
            return {
                mensaje: "No tienes tareas registradas en este momento."
            };
        }

        // Agrupar tareas por materia
        const tasksBySubject = tasks.reduce((acc, task) => {
            if (!acc[task.subject_]) {
                acc[task.subject_] = [];
            }
            acc[task.subject_].push({
                nombre: task.name_task,
                estado: task.status_task,
                fecha_entrega: task.end_date_task,
                instrucciones: task.instructions_task,
                calificacion: task.grade_,
                observaciones: task.observations_profe,
                url_archivo: task.file_url_task || 'No disponible'
            });
            return acc;
        }, {});

        return {
            total_tareas: tasks.length,
            periodo: tasks[0].period_,
            estudiante: tasks[0].student_,
            materias: Object.keys(tasksBySubject).map(subject => ({
                nombre_materia: subject,
                tareas: tasksBySubject[subject]
            }))
        };
    }

    calculatePerformanceScore(responseTime) {
        // Escala de rendimiento: 100 (excelente) a 0 (pobre)
        if (responseTime < 1000) return 100;
        if (responseTime < 2000) return 90;
        if (responseTime < 3000) return 80;
        if (responseTime < 4000) return 70;
        if (responseTime < 5000) return 60;
        return 50;
    }

    calculateCostEstimate(totalTokens) {
        // Costos aproximados por token
        const costPerToken = 0.00002; // Ajustar según el modelo y precios actuales
        return totalTokens * costPerToken;
    }

    // Metodo para limpiar el hilo de un usuario si es necesario
    async clearUserThread(phoneNumber) {
        try {
            if (this.userThreads.has(phoneNumber)) {
                const threadId = this.userThreads.get(phoneNumber);
                await client.beta.threads.del(threadId);
                this.userThreads.delete(phoneNumber);
                console.log(`🧹 Hilo limpiado para: ${phoneNumber}`);
            }
        } catch (error) {
            console.error(`❌ Error limpiando hilo de ${phoneNumber}:`, error);
        }
    }
}



// Exportar una única instancia del servicio
module.exports = new AssistantService();