const prisma = require('../../lib/prisma');

class ConversationService {
    async getOrCreateConversation(phoneNumber, threadId) {
        try {
            // Buscar conversación activa existente
            let conversation = await prisma.conversation.findFirst({
                where: {
                    phone_number: phoneNumber,
                    is_active: true
                }
            });

            // Si no existe, crear nueva conversación
            if (!conversation) {
                conversation = await prisma.conversation.create({
                    data: {
                        thread_id: threadId,
                        is_active: true,
                        context_depth: 4,
                        user: {
                            connect: {
                                phone_number: phoneNumber
                            }
                        }
                    }
                });
                console.log(`✅ Nueva conversación creada para: ${phoneNumber}`);
            }

            return conversation;

        } catch (error) {
            console.error('❌ Error en getOrCreateConversation:', error);
            throw error;
        }
    }

    async getConversationContext(phoneNumber, limit = 4) {
        try {
            const conversation = await prisma.conversation.findFirst({
                where: {
                    phone_number: phoneNumber,
                    is_active: true
                },
                include: {
                    messages: {
                        orderBy: { timestamp: 'desc' },
                        take: limit,
                        include: {
                            generated_images: true
                        }
                    }
                }
            });

            return conversation?.messages || [];
        } catch (error) {
            console.error('❌ Error en getConversationContext:', error);
            throw error;
        }
    }

    async updateConversation(conversationId, data) {
        try {
            return await prisma.conversation.update({
                where: {
                    conversation_id: conversationId
                },
                data: {
                    ...data,
                    last_interaction_date: new Date()
                }
            });
        } catch (error) {
            console.error('❌ Error en updateConversation:', error);
            throw error;
        }
    }

    // En conversationService.js, añadir este metodo
    async getConversationById(conversationId) {
        try {
            return await prisma.conversation.findUnique({
                where: {
                    conversation_id: conversationId
                }
            });
        } catch (error) {
            console.error('❌ Error en getConversationById:', error);
            throw error;
        }
    }
}

module.exports = new ConversationService();