const prisma = require('../../lib/prisma');

class MessageService {

// services/database/messageService.js

    async saveMessage(data) {
        try {
            console.log('📥 Datos recibidos en saveMessage:', {
                conversationId: data.conversationId,
                phoneNumber: data.phoneNumber,
                messageType: data.messageType
            });

            const message = await prisma.message.create({
                data: {
                    // Conectar con la conversación
                    conversation: {
                        connect: {
                            conversation_id: data.conversationId
                        }
                    },
                    // Conectar con el usuario
                    user: {
                        connect: {
                            phone_number: data.phoneNumber
                        }
                    },
                    // Campos del mensaje
                    message_type: data.messageType,
                    content: data.content,
                    tokens_used: data.tokensUsed || 0,
                    model_used: data.modelUsed,
                    completion_tokens: data.completionTokens || 0,
                    prompt_tokens: data.promptTokens || 0,
                    total_tokens: data.totalTokens || 0,
                    response_time: data.responseTime
                }
            });

            return message;
        } catch (error) {
            console.error('❌ Error en saveMessage:', error);
            throw error;
        }
    }

    async getRecentMessages(conversationId, limit = 4) {
        try {
            if (!conversationId) {
                console.log('No hay conversation_id proporcionado');
                return [];
            }

            console.log(`🔍 Buscando mensajes para conversación: ${conversationId}`);

            const messages = await prisma.message.findMany({
                where: {
                    conversation_id: conversationId
                },
                orderBy: {
                    timestamp: 'desc'
                },
                take: limit,
                select: {
                    message_id: true,
                    conversation_id: true,
                    phone_number: true,
                    message_type: true,
                    content: true,
                    timestamp: true,
                    tokens_used: true,
                    model_used: true,
                    completion_tokens: true,
                    prompt_tokens: true,
                    total_tokens: true,
                    response_time: true,
                    generated_images: {
                        select: {
                            image_id: true,
                            prompt_used: true,
                            image_url: true,
                            model_used: true,
                            generation_status: true
                        }
                    }
                }
            });

            console.log(`📝 Encontrados ${messages.length} mensajes`);
            return messages;

        } catch (error) {
            console.error('❌ Error en getRecentMessages:', error);
            console.error('Detalles del error:', JSON.stringify(error, null, 2));
            return [];
        }
    }


    async getConversationContext(conversationId, limit = 5) {
        try {
            if (!conversationId) {
                console.log('No hay conversation_id proporcionado');
                return [];
            }

            const messages = await prisma.message.findMany({
                where: {
                    conversation_id: conversationId
                },
                orderBy: {
                    timestamp: 'desc' // Orden cronológico
                },
                take: limit, // limite de mensajes recuperados por defecto es 5
                select: {
                    message_type: true,
                    content: true,
                    timestamp: true
                }
            });

            return messages.map(msg => ({
                role: msg.message_type === 'user' ? 'user' : 'assistant',
                content: msg.content,
                timestamp: msg.timestamp
            }));
        } catch (error) {
            console.error('❌ Error en getConversationContext:', error);
            return [];
        }
    }

    async getMessagesByUser(phoneNumber, limit = 10) {
        try {
            return await prisma.message.findMany({
                where: {
                    phone_number: phoneNumber
                },
                orderBy: {
                    timestamp: 'desc'
                },
                take: limit,
                include: {
                    generated_images: true,
                    conversation: true
                }
            });
        } catch (error) {
            console.error('❌ Error en getMessagesByUser:', error);
            return [];
        }
    }

    async updateMessage(messageId, data) {
        try {
            return await prisma.message.update({
                where: {
                    message_id: messageId
                },
                data: {
                    ...data,
                    timestamp: new Date()
                }
            });
        } catch (error) {
            console.error('❌ Error en updateMessage:', error);
            throw error;
        }
    }

    async getMessageCount(conversationId) {
        try {
            return await prisma.message.count({
                where: {
                    conversation_id: conversationId
                }
            });
        } catch (error) {
            console.error('❌ Error en getMessageCount:', error);
            return 0;
        }
    }
}

module.exports = new MessageService();