const { Pool } = require('pg');

class TaskService {
    constructor() {
        this.academicPool = new Pool({
            connectionString: process.env.ACADEMIC_DATABASE_URL
        });
    }

    async getStudentInfo(phoneNumber) {
        try {
            const query = `
                SELECT DISTINCT
                    concat(c.last_name1,' ',c.last_name2, ' ',c.first_name) AS nombre,
                    g.name AS semester,
                    h.name AS parallel,
                    i.name AS carrera
                FROM student AS c 
                INNER JOIN enrollment AS e ON e.student_id = c.id AND e.status = 'ACTIVE'
                INNER JOIN syllabus AS a ON a.id = e.syllabus_id 
                INNER JOIN semester AS g ON g.id = a.semester_id
                INNER JOIN parallel AS h ON h.id = a.parallel_id
                INNER JOIN subject AS i ON i.id = a.subject_id
                WHERE c.phone = $1
                LIMIT 1
            `;

            const result = await this.academicPool.query(query, [phoneNumber]);

            if (result.rows.length === 0) {
                console.log(`ℹ️ No se encontró información para el estudiante: ${phoneNumber}`);
                return {
                    success: false,
                    message: "Estudiante no encontrado"
                };
            }

            console.log(`✅ Información del estudiante recuperada:`, result.rows[0]);
            return {
                success: true,
                student: result.rows[0]
            };
        } catch (error) {
            console.error('❌ Error en getStudentInfo:', error);
            return {
                success: false,
                message: "Error al obtener información del estudiante"
            };
        }
    }

    async getStudentTasks(phoneNumber) {
        try {
            const query = `
                SELECT 
                    concat(c.last_name1,' ',c.last_name2, ' ',c.first_name) AS student_, 
                    f.name AS period_,
                    i.name AS subject_,
                    g.name AS semester_,
                    h.name AS parallel_,
                    k.name AS name_task, 
                    k.instructions AS instructions_task, 
                    k.start_date AS start_date_task,
                    k.end_date AS end_date_task, 
                    k.file_url AS file_url_task, 
                    j.status AS status_task,
                    CASE 
                        WHEN j.status = 'PENDING' THEN 'Tarea aún no subida'
                        ELSE COALESCE(j.submission_date::text, 'Sin fecha de entrega')
                    END AS submission_date,
                    CASE 
                        WHEN j.status = 'PENDING' THEN 'Sin observaciones - Tarea pendiente'
                        WHEN j.status = 'SUBMITTED' THEN 'Sin observaciones - En revisión'
                        ELSE COALESCE(j.observations, 'Sin observaciones')
                    END AS observations_profe,
                    CASE 
                        WHEN j.status = 'PENDING' THEN 'Sin calificación - Tarea pendiente'
                        WHEN j.status = 'SUBMITTED' THEN 'Sin calificación - En revisión'
                        ELSE COALESCE(j.grade::text, 'Sin calificación')
                    END AS grade_
                FROM student AS c 
                INNER JOIN enrollment AS e ON e.student_id = c.id AND e.status = 'ACTIVE'
                INNER JOIN syllabus AS a ON a.id = e.syllabus_id 
                INNER JOIN period AS f ON f.id = a.period_id AND f.status = 'A'
                INNER JOIN subject AS i ON i.id = a.subject_id
                INNER JOIN semester AS g ON g.id = a.semester_id
                INNER JOIN parallel AS h ON h.id = a.parallel_id
                INNER JOIN student_task AS j ON j.enrollment_id = e.id
                INNER JOIN task AS k ON k.id = j.task_id AND k.syllabus_id = a.id
                WHERE c.phone = $1
            `;

            const result = await this.academicPool.query(query, [phoneNumber]);

            if (result.rows.length === 0) {
                return {
                    success: false,
                    message: "No se encontraron tareas para este estudiante"
                };
            }
            return {
                success: true,
                tasks: result.rows,
                totalTasks: result.rows.length
            };
        } catch (error) {
            console.error('❌ Error en getStudentTasks:', error);
            return {
                success: false,
                message: "Error al obtener las tareas",
                error: error.message
            };
        }
    }

}

module.exports = new TaskService();